import i18next from "i18next";
import { GetServerSideProps } from "next";
import { parseCookies } from "nookies";
import { ChangeEvent, useRef, useState } from "react";
import useFile from "../../core/hooks/useFile";
import { ImageError } from "../../core/utils/exceptions";
import { Util } from "../../core/utils/util";
import { Toast } from "primereact/toast";
import { Formik, Form } from "formik";
import * as Yup from "yup";
import "yup-phone";
import YupPassword from "yup-password";
import Password from "../../app/components/SvgIcons/password";
import Mail from "../../app/components/SvgIcons/mail";
import Input from "../../app/components/CustomInput";
import Button from "../../app/components/Button";
import { UserRegisterDto } from "../../core/models/interfaces/User/UserRegisterDto";
import useAuth from "../../core/hooks/useAuth";
import router from "next/router";
import { masks } from "../../core/utils/masks";
import FormLoading from "../../app/components/FormLoading";
import { Checkbox } from "primereact/checkbox";
import Cellphone from "../../app/components/SvgIcons/cellphone";
import UserIcon from "../../app/components/SvgIcons/user";
import { User } from "../../core/models/interfaces/User/User";
import { serverSideTranslations } from "next-i18next/serverSideTranslations";

import { Wrapper, RegisterImage, Title, TextContainer, Text, TextLink, ButtonContainer } from "./styles";
import Head from "next/head";
import { useTranslation } from "next-i18next";

YupPassword(Yup);

export default function Register() {
  const inputRef = useRef<any>();
  const toastRef = useRef<any>();
  const { t } = useTranslation("common");

  const { getArchiveByHtmlInput, archiveInfos } = useFile("image");
  const { registerUser } = useAuth();
  const [buttonLoading, setButtonLoading] = useState(false);
  const [wantToBeASupplier, setWantToBeSupplier] = useState(false);

  const handleOnImageChnage = (event: ChangeEvent<HTMLInputElement>) => {
    try {
      getArchiveByHtmlInput(event);
    } catch (error) {
      if (error instanceof ImageError) {
        Util.showErrorToast(toastRef, error.message);
      } else {
        Util.showErrorToast(toastRef, t("onUploadImageError"));
      }
    }
  };

  const passwordConfirmationValidation = (senha: string) => {
    if (senha) {
      return true;
    }
    return false;
  };

  const initialValues = {
    name: "",
    email: "",
    cellPhone: "",
    password: "",
    repeatPassword: "",
  };

  interface registerInitialValues {
    name: string;
    email: string;
    cellPhone: string;
    password: string;
    repeatPassword: string;
  }

  const onSubmit = async (values: registerInitialValues) => {
    if (buttonLoading) return;

    let body: UserRegisterDto = {
      nome: values.name,
      email: values.email,
      celular: values.cellPhone,
      senha: values.password,
      perfilAtual: wantToBeASupplier ? "E" : "C",
      foto: archiveInfos?.base64
        ? {
            bytes: archiveInfos?.base64,
            nome: archiveInfos?.nome as string,
            contentType: archiveInfos?.contentType as string,
          }
        : null,
      //default, vamo pegar a do sistema dps
      linguagemAtual: "pt-BR",
    };

    setButtonLoading(true);

    try {
      await registerUser(body);
      Util.showSucessToast(toastRef, t("successAccount"));
      router.replace("/home?page=1");
    } catch (error) {
      Util.showErrorToast(toastRef, t("registerError"));
    } finally {
      setButtonLoading(false);
    }
  };

  const validationSchema = Yup.object({
    name: Yup.string().required(t("requiredField")),
    email: Yup.string().email(t("invalidEmail")).required(t("requiredField")),
    cellPhone: Yup.string().phone("BR", false, t("invalidCellphone")).required(t("requiredField")),
    password: Yup.string()
      .password()
      .minUppercase(0)
      .minLowercase(0)
      .minSymbols(0)
      .minNumbers(0)
      .min(6, t("passwordLength"))
      .max(16, t("passwordLength")),
    repeatPassword: Yup.string().when("password", {
      is: (value: any) => value && value.length > 0,
      then: Yup.string()
        .oneOf([Yup.ref("password")], t("passwordsMatch"))
        .required(t("requiredField")),
      otherwise: Yup.string().notRequired(),
    }),
  });

  return (
    <Wrapper>
      <Head>
        <title>Relty - {t("createAccount")}</title>
      </Head>
      <FormLoading show={buttonLoading} />
      {/* @ts-ignore */}
      <Toast ref={toastRef} position="top-center" />
      <Title>{t("createAccount")}</Title>

      <RegisterImage
        url={archiveInfos?.uri || "/assets/perfil-default.png"}
        onClick={() => {
          inputRef.current.click();
        }}
      >
        <input
          ref={inputRef}
          type="file"
          onChange={(event) => {
            handleOnImageChnage(event);
          }}
        ></input>
      </RegisterImage>

      <Formik
        initialValues={initialValues}
        validationSchema={validationSchema}
        onSubmit={(values) => onSubmit(values)}
        validateOnChange
      >
        {(formik) => {
          return (
            <Form>
              <Input
                id="name"
                name="name"
                label={t("name")}
                placeholder="João Alberto"
                error={formik.errors.name && formik.touched.name}
                onBlur={(event: any) => {
                  if (passwordConfirmationValidation(formik.values.password)) {
                    formik.setFieldTouched("repeatPassword", true, true);
                  }
                }}
                icon={<UserIcon width={20} height={20}></UserIcon>}
              />
              <Input
                id="email"
                name="email"
                label={t("email")}
                placeholder="example@gmail.com"
                error={formik.errors.email && formik.touched.email}
                onBlur={(event: any) => {
                  if (passwordConfirmationValidation(formik.values.password)) {
                    formik.setFieldTouched("repeatPassword", true, true);
                  }
                }}
                icon={<Mail width={20} height={20}></Mail>}
              />
              <Input
                id="cellPhone"
                name="cellPhone"
                label={t("cellPhone")}
                placeholder="(99) 9 9999-9999"
                onInput={(event: any) => {
                  event.nativeEvent.target.value = masks.brCellphone(event.nativeEvent.target.value);
                }}
                error={formik.errors.cellPhone && formik.touched.cellPhone}
                onBlur={(event: any) => {
                  if (passwordConfirmationValidation(formik.values.password)) {
                    formik.setFieldTouched("repeatPassword", true, true);
                  }
                }}
                icon={<Cellphone width={20} height={20}></Cellphone>}
              />

              <Input
                id="password"
                name="password"
                label={t("password")}
                type="password"
                placeholder="******"
                error={formik.errors.password && formik.touched.password}
                icon={<Password width={20} height={20}></Password>}
              />

              <Input
                id="repeatPassword"
                name="repeatPassword"
                label={t("repeatPassword")}
                type="password"
                placeholder="******"
                error={formik.errors.repeatPassword && formik.touched.repeatPassword}
                icon={<Password width={20} height={20}></Password>}
              />
              <div className="p-field-checkbox">
                {/* @ts-ignore */}
                <Checkbox
                  inputId="binary"
                  checked={wantToBeASupplier}
                  onChange={(e) => setWantToBeSupplier(e.checked)}
                />
                <label htmlFor="binary">{t("wantToBeSupplier")}</label>
              </div>

              <TextContainer>
                <Text>
                  {t("youAcceptTerms")}
                  <br />

                  <TextLink href="https://reltyapp.com.br/fornecedores/" target="_blank">
                    {t("supplierTerms")}
                  </TextLink>
                  <br />
                  <TextLink href="https://reltyapp.com.br/termos-usuario/" target="_blank">
                    {t("usersTerms")}
                  </TextLink>
                </Text>
              </TextContainer>

              <ButtonContainer>
                <Button text={t("createAccount")} disabled={!formik.isValid}></Button>
              </ButtonContainer>
            </Form>
          );
        }}
      </Formik>
    </Wrapper>
  );
}

export const getServerSideProps: GetServerSideProps = async (ctx) => {
  const { res, resolvedUrl } = ctx;
  const token = parseCookies(ctx).access_token;
  const savedUser = parseCookies(ctx).user;

  if (token) {
    Util.sendToHomePage(res);
  }

  let language = parseCookies(ctx).language || "pt";
  if (ctx.locale != language && language) {
    return Util.redirectToSelectedLanguage(ctx, resolvedUrl);
  }

  if (savedUser) {
    let user = JSON.parse(savedUser) as User;
    return Util.sendToAnyPage(res, `/profiles/${user.perfilAtual == "E" ? "supplier" : "user"}`);
  }

  return {
    props: {
      ...(await serverSideTranslations(ctx.locale as string, ["common"])),
    },
  };
};
