import { GetServerSideProps } from "next";
import router from "next/router";
import { useEffect, useState } from "react";
import { AdventureDetailsDto } from "../../core/models/interfaces/Adventure/AdventureDetailsDto";
import { getApi } from "../../core/services/apiServerSide";
import { Util } from "../../core/utils/util";
import { Wrapper, Container, Icon } from "./styles";
import Head from "next/head";
import { parseCookies } from "nookies";
import { serverSideTranslations } from "next-i18next/serverSideTranslations";
import Button from "../../app/components/Button";
import { useTranslation } from "react-i18next";
import { useTheme } from "styled-components";

interface RedirectProps {
  adventureId: number;
  slug: string;
  imageUrl: string;
}

export default function Redirect({ adventureId, slug, imageUrl }: RedirectProps) {
  const { t } = useTranslation("common");
  const theme = useTheme();
  const [isMobile, setIsMobile] = useState(false);

  useEffect(() => {
    setIsMobile(Util.isMobile());
    //Caso for um navegador jogamos pra própria pagina da aventura dentro do site.
    if (!Util.isMobile()) {
      router.replace(`/adventure-details/${adventureId}/${slug}`);
      return;
    }
    setTimeout(function () {
      //Faz apenas se for iOS.
      if (Util.isMobile() && Util.isAndroid()) {
        window.location.href = "https://relty.app.link/";
      }
      return;
    }, 400);
    window.location.href = `relty://?aventuraid=${adventureId}`;
  }, []);

  const getApp = () => {
    window.location.href = "https://relty.app.link/";
  };

  const goToHome = () => {
    router.replace("/home");
  };

  return (
    <Wrapper>
      <Head>
        <meta property="og:image" content={imageUrl}></meta>
      </Head>

      {isMobile && (
        <Container>
          <Icon />
          <span>
            <Button onClick={getApp} text={t("getApp")}></Button>
          </span>
          <span>
            <Button onClick={goToHome} color={theme.secondaryBackground} text={t("home")}></Button>
          </span>
        </Container>
      )}
    </Wrapper>
  );
}

export const getServerSideProps: GetServerSideProps<RedirectProps> = async (ctx) => {
  console.log('passei aqui ===>', ctx)
  const { query } = ctx;
  const { adventureId: _adventureId } = query;
  const adventureId = Number(_adventureId);

  const api = getApi(ctx);

  if (!adventureId || isNaN(adventureId) || adventureId < 1) {
    return { notFound: true };
  }

  const adventureDetail = await api.get<AdventureDetailsDto>(`/WS/Aventura/Completa/${adventureId}`);

  return {
    props: {
      ...(await serverSideTranslations(ctx.locale as string, ["common"])),
      adventureId,
      slug: adventureDetail.data.slug,
      imageUrl: adventureDetail.data?.imagens[0]?.link,
    },
  };
};
