import React, { useState, useEffect, useCallback, useRef, ChangeEvent } from "react";
import { Formik, Form } from "formik";
import * as Yup from "yup";
import "yup-phone";
import YupPassword from "yup-password";
import { Wrapper, RegisterImage, InputGroup, Title, InputContainer, ButtonContainer } from "./styles";
import { UserEditDto } from "../../../core/models/interfaces/User/UserEditDto";
import i18next from "i18next";
import Input from "../../../app/components/CustomInput";
import Mail from "../../../app/components/SvgIcons/mail";
import Password from "../../../app/components/SvgIcons/password";
import useUser from "../../../core/hooks/useUser";
import { Toast } from "primereact/toast";
import { Util } from "../../../core/utils/util";
import { ImageError } from "../../../core/utils/exceptions";
import useFile from "../../../core/hooks/useFile";
import { masks } from "../../../core/utils/masks";
import Head from "next/head";
import { parseCookies } from "nookies";
import { GetServerSideProps } from "next";
import { User } from "../../../core/models/interfaces/User/User";
import Button from "../../../app/components/Button";
import FormLoading from "../../../app/components/FormLoading";
YupPassword(Yup);
import { useUserContext } from "../../../core/contexts/UserContext";
import UserIcon from "../../../app/components/SvgIcons/user";
import Cellphone from "../../../app/components/SvgIcons/cellphone";
import { useTheme } from "styled-components";
import { HiDocumentText } from "react-icons/hi";
import { serverSideTranslations } from "next-i18next/serverSideTranslations";
import { useTranslation } from "next-i18next";

function SupplierProfile() {
  const { getDataEditing, saveDataEditing } = useUser();
  const { t } = useTranslation("common");
  const { updateUser } = useUserContext();
  const [userEditDto, setUserEditDto] = useState<UserEditDto>({} as UserEditDto);
  const { getArchiveByHtmlInput, archiveInfos } = useFile("image");
  const [loading, setLoading] = useState(true);
  const theme = useTheme();

  const inputRef = useRef<any>();
  const toastRef = useRef<any>();

  const validationSchema = Yup.object({
    usuario: Yup.object({
      nome: Yup.string().required(t("requiredField")),
      email: Yup.string().email(t("invalidEmail")).required(t("requiredField")),
      celular: Yup.string().phone("BR", false, t("invalidCellphone")).required(t("requiredField")),
    }),
    pessoa: Yup.object({
      cpfCnpj: Yup.string().required(t("requiredField")),
    }),

    senha: Yup.string()
      .password()
      .minUppercase(0)
      .minLowercase(0)
      .minSymbols(0)
      .minNumbers(0)
      .min(6, t("passwordLength"))
      .max(16, t("passwordLength")),
    senhaConfirmar: Yup.string().when("senha", {
      is: (value: any) => value && value.length > 0,
      then: Yup.string()
        .oneOf([Yup.ref("senha")], t("passwordsMatch"))
        .required(t("requiredField")),
      otherwise: Yup.string().notRequired(),
    }),
  });

  const passwordConfirmationValidation = (senha: string) => {
    if (senha) {
      return true;
    }
    return false;
  };

  const handleOnImageChnage = (event: ChangeEvent<HTMLInputElement>) => {
    try {
      getArchiveByHtmlInput(event);
    } catch (error) {
      if (error instanceof ImageError) {
        Util.showErrorToast(toastRef, error.message);
      } else {
        Util.showErrorToast(toastRef, t("onUploadImageError"));
      }
    }
  };

  const getUserEditDto = useCallback(async () => {
    const response = await getDataEditing();
    setUserEditDto(response.data);
    setLoading(false);
  }, []);

  const onSubmit = async (user: any, onSubmitProps: any) => {
    try {
      setLoading(true);

      if (user.senha) {
        user.usuario.senha = user.senha;
      }
      if (archiveInfos?.base64 != null) {
        user.foto = {
          bytes: archiveInfos?.base64,
          nome: archiveInfos?.nome,
          contentType: archiveInfos?.contentType,
        };
      }

      await saveDataEditing(user);
      onSubmitProps.setSubmitting(false);
      updateUser();
      Util.showSucessToast(toastRef, t("saveDataSuccess"));
    } catch (error) {
      Util.showErrorToast(toastRef, t("saveDataError"));
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    getUserEditDto();
  }, [getUserEditDto]);

  return (
    <Wrapper>
      <Head>
        <title>Relty - {t("myProfile")}</title>
      </Head>
      <FormLoading show={loading} />
      {/* @ts-ignore */}
      <Toast ref={toastRef} position="top-center" />
      <Title>{t("editProfile")}</Title>

      <RegisterImage
        url={archiveInfos?.uri || userEditDto.foto?.url || "/assets/perfil-default.png"}
        onClick={() => {
          inputRef.current.click();
        }}
      >
        <input
          ref={inputRef}
          type="file"
          onChange={(event) => {
            handleOnImageChnage(event);
          }}
        ></input>
      </RegisterImage>

      <Formik
        initialValues={userEditDto}
        validationSchema={validationSchema}
        onSubmit={onSubmit}
        enableReinitialize
        validateOnChange
      >
        {(formik: any) => {
          return (
            <Form>
              <InputGroup inputs={2}>
                <InputContainer>
                  <Input
                    id="nome"
                    name="usuario.nome"
                    label={t("name")}
                    placeholder={t("placeholder_responsavel")}
                    error={formik.errors.usuario?.nome && formik.errors.usuario?.nome}
                    icon={<UserIcon width={20} height={20}></UserIcon>}
                  />
                </InputContainer>
                <InputContainer>
                  <Input
                    id="email"
                    name="usuario.email"
                    label={t("email")}
                    placeholder="example@gmail.com"
                    error={formik.errors.usuario?.email && formik.touched.usuario?.email}
                    icon={<Mail width={20} height={20}></Mail>}
                  />
                </InputContainer>
              </InputGroup>

              <InputGroup inputs={2}>
                <InputContainer>
                  <Input
                    id="celular"
                    name="usuario.celular"
                    label={t("cellPhone")}
                    placeholder={t("cellPhonePlaceholder")}
                    onInput={(event: any) => {
                      event.nativeEvent.target.value = masks.brCellphone(event.nativeEvent.target.value);
                    }}
                    error={formik.errors.usuario?.celular && formik.touched.usuario?.celular}
                    icon={<Cellphone width={20} height={20}></Cellphone>}
                  />
                </InputContainer>
                <InputContainer>
                  <Input
                    id="cpfCnpj"
                    name="pessoa.cpfCnpj"
                    label={t("cpfCnpj")}
                    placeholder={t("cpfCnpj")}
                    onInput={(event: any) => {
                      event.nativeEvent.target.value = masks.cpfCnpj(event.nativeEvent.target.value);
                    }}
                    error={formik.errors.pessoa?.cpfCnpj && formik.touched.pessoa?.cpfCnpj}
                    icon={<HiDocumentText size={22} color={theme.primaryBackground} />}
                  />
                </InputContainer>
              </InputGroup>

              <InputGroup inputs={2}>
                <InputContainer>
                  <Input
                    id="senha"
                    type="password"
                    name="senha"
                    label={t("password")}
                    placeholder="******"
                    error={formik.errors.senha && formik.touched.senha}
                    icon={<Password width={20} height={20}></Password>}
                    onBlur={() => {
                      if (passwordConfirmationValidation(formik.values.senha)) {
                        formik.setFieldTouched("senhaConfirmar", true, true);
                      }
                    }}
                  />
                </InputContainer>
                <InputContainer>
                  <Input
                    id="senhaConfirmar"
                    type="password"
                    name="senhaConfirmar"
                    label={t("repeatPassword")}
                    placeholder="******"
                    error={formik.errors.senhaConfirmar && formik.touched.senhaConfirmar}
                    icon={<Password width={20} height={20}></Password>}
                  />
                </InputContainer>
              </InputGroup>

              <ButtonContainer>
                <Button
                  type="submit"
                  text={t("submit")}
                  disabled={formik.isSubmitting || !formik.isValid}
                  className="p-mt-2"
                />
              </ButtonContainer>
            </Form>
          );
        }}
      </Formik>
    </Wrapper>
  );
}

export default SupplierProfile;

export const getServerSideProps: GetServerSideProps = async (ctx) => {
  const { res, resolvedUrl } = ctx;
  const token = parseCookies(ctx).access_token;
  const savedUser = parseCookies(ctx).user;

  if (!token) {
    return Util.sendToHomePage(res, undefined);
  }
  let language = parseCookies(ctx).language || "pt";
  if (ctx.locale != language && language) {
    return Util.redirectToSelectedLanguage(ctx, resolvedUrl);
  }

  if (savedUser) {
    let user = JSON.parse(savedUser) as User;
    if (user.perfilAtual == "E") {
      return Util.sendToAnyPage(res, "/profiles/supplier");
    }
  }

  return {
    props: {
      ...(await serverSideTranslations(ctx.locale as string, ["common"])),
    },
  };
};
