import React, { useState, useEffect, useCallback, useRef, ChangeEvent } from "react";
import { Formik, Form } from "formik";
import * as Yup from "yup";
import "yup-phone";
import YupPassword from "yup-password";
import {
  ButtonContainer,
  Container,
  Wrapper,
  InputContainer,
  InputGroup,
  RegisterImage,
  Title,
  ContainerText,
  Text,
} from "./styles";
import useEstablishment from "../../../core/hooks/useEstablishment";
import { UserEditDto } from "../../../core/models/interfaces/User/UserEditDto";
import i18next from "i18next";
import Input from "../../../app/components/CustomInput";
import Mail from "../../../app/components/SvgIcons/mail";
import Password from "../../../app/components/SvgIcons/password";
import useBanks from "../../../core/hooks/useBanks";
import { Toast } from "primereact/toast";
import { Util } from "../../../core/utils/util";
import { ImageError } from "../../../core/utils/exceptions";
import useFile from "../../../core/hooks/useFile";
import Cellphone from "../../../app/components/SvgIcons/cellphone";
import UserIcon from "../../../app/components/SvgIcons/user";
import { masks } from "../../../core/utils/masks";
import { GetServerSideProps } from "next";
import { User } from "../../../core/models/interfaces/User/User";
import Head from "next/head";
import FormLoading from "../../../app/components/FormLoading";
import Button from "../../../app/components/Button";
YupPassword(Yup);
import { useUserContext } from "../../../core/contexts/UserContext";
import { HiDocumentText } from "react-icons/hi";
import { CgDetailsMore } from "react-icons/cg";
import { AiFillBank } from "react-icons/ai";
import { BiTransfer } from "react-icons/bi";
import { useTheme } from "styled-components";
import { parseCookies } from "nookies";
import { serverSideTranslations } from "next-i18next/serverSideTranslations";
import { useTranslation } from "next-i18next";

function SupplierProfile() {
  const { getData } = useBanks();
  const { t } = useTranslation("common");
  const { getDataEditing, saveDataEditing } = useEstablishment();
  const { updateUser } = useUserContext();

  const [userEditDto, setUserEditDto] = useState<UserEditDto>({} as UserEditDto);
  const [banks, setBanks] = useState<any[]>([]);
  const [filteredBanks, setFilteredBanks] = useState<any[]>([]);
  const inputRef = useRef<any>();
  const toastRef = useRef<any>();
  const { getArchiveByHtmlInput, archiveInfos } = useFile("image");
  const [loading, setLoading] = useState(true);
  const theme = useTheme();

  const validationSchema = Yup.object({
    usuario: Yup.object({
      nome: Yup.string().required(t("requiredField")),
      email: Yup.string().email(t("invalidEmail")).required(t("requiredField")),
      celular: Yup.string().phone("BR", false, t("invalidCellphone")).required(t("requiredField")),
    }),
    pessoa: Yup.object({
      email: Yup.string().email(t("invalidEmail")).required(t("requiredField")),
      cpfCnpj: Yup.string().required(t("requiredField")),
      contaBancaria: Yup.object({
        codigoBanco: Yup.string().required(t("requiredField")),
        contaBancaria: Yup.string().required(t("requiredField")),
        digitoContaBancaria: Yup.string().required(t("requiredField")),
        agenciaBancaria: Yup.string().required(t("requiredField")),
        intervaloDeTransferencia: Yup.string().required(t("requiredField")),
        cpfCnpjResponsavel: Yup.string().required(t("requiredField")),
        nomeRazaoSocial: Yup.string().required(t("requiredField")),
      }),
    }),
    estabelecimento: Yup.object({
      descricaoComercial: Yup.string().required(t("requiredField")),
    }),
    senha: Yup.string()
      .password()
      .minUppercase(0)
      .minLowercase(0)
      .minSymbols(0)
      .minNumbers(0)
      .min(6, t("passwordLength"))
      .max(16, t("passwordLength")),
    senhaConfirmar: Yup.string().when("senha", {
      is: (value: any) => value && value.length > 0,
      then: Yup.string()
        .oneOf([Yup.ref("senha")], t("passwordsMatch"))
        .required(t("requiredField")),
      otherwise: Yup.string().notRequired(),
    }),
  });

  const validateEmail = (value: any) => {
    let error;
    if (!value) {
      error = t("requiredField");
    }
    return error;
  };

  const validateEmailUser = (value: any) => {
    let error;
    if (!value) {
      error = t("requiredField");
    }
    return error;
  };

  const dropdownOptions = [
    { key: t("daily"), value: "DAILY" },
    { key: t("weekly"), value: "WEEKLY" },
    { key: t("monthly"), value: "MONTHLY" },
  ];

  const passwordConfirmationValidation = (senha: string) => {
    if (senha) {
      return true;
    }
    return false;
  };

  const handleOnImageChnage = (event: ChangeEvent<HTMLInputElement>) => {
    try {
      getArchiveByHtmlInput(event);
    } catch (error) {
      if (error instanceof ImageError) {
        Util.showErrorToast(toastRef, error.message);
      } else {
        Util.showErrorToast(toastRef, t("onUploadImageError"));
      }
    }
  };

  const getbancosFormatados = useCallback(async () => {
    let bancosFormatados: any[] = [];
    let bancos: any[] = [];
    const response = await getData();
    bancos = response.data;

    bancos.forEach((banco: any) => {
      let item = { key: banco.Code + " - " + banco.Name, value: banco.Code };
      bancosFormatados.push(item);
    });

    bancosFormatados = bancosFormatados.sort(function (a: any, b: any) {
      if (a.value > b.value) {
        return 1;
      }
      if (a.value < b.value) {
        return -1;
      }
      return 0;
    });

    setBanks(bancosFormatados);
  }, []);

  const searchBanks = (event: { query: string }) => {
    setTimeout(() => {
      let _filteredBanks: any[] = [];
      if (!event.query.trim().length) {
        _filteredBanks = [...banks];
      } else {
        _filteredBanks = banks.filter((bank: any) => {
          return bank.name.toLowerCase().includes(event.query.toLowerCase());
        });
      }
      setFilteredBanks(_filteredBanks);
    }, 250);
  };

  const getUserEditDto = useCallback(async () => {
    const response = await getDataEditing();
    setUserEditDto(response.data);
    setLoading(false);
  }, []);

  useEffect(() => {
    getbancosFormatados();
  }, [getbancosFormatados]);

  useEffect(() => {
    getUserEditDto();
  }, [getUserEditDto]);

  const onSubmit = async (user: any, onSubmitProps: any) => {
    if (user.senha) {
      user.usuario.senha = user.senha;
    }
    setLoading(true);
    try {
      if (archiveInfos?.base64 != null) {
        user.foto = {
          bytes: archiveInfos?.base64,
          nome: archiveInfos?.nome,
          contentType: archiveInfos?.contentType,
        };
      }
      await saveDataEditing(user);
      onSubmitProps.setSubmitting(false);
      updateUser();
      Util.showSucessToast(toastRef, t("saveDataSuccess"));
    } catch (error) {
      Util.showErrorToast(toastRef, t("saveDataError"));
    } finally {
      setLoading(false);
    }
  };

  function getPlaceholder(name: string) {
    return t(name);
  }

  return (
    <Wrapper>
      <Head>
        <title>Relty - {t("myProfile")}</title>
      </Head>
      <FormLoading show={loading} />
      {/* @ts-ignore */}
      <Toast ref={toastRef} position="top-center" />
      <Title>{t("editProfile")}</Title>

      <RegisterImage
        url={archiveInfos?.uri || userEditDto.foto?.url || "/assets/perfil-default.png"}
        onClick={() => {
          inputRef.current.click();
        }}
      >
        <input
          ref={inputRef}
          type="file"
          onChange={(event) => {
            handleOnImageChnage(event);
          }}
        ></input>
      </RegisterImage>

      <Formik
        initialValues={userEditDto}
        validationSchema={validationSchema}
        onSubmit={onSubmit}
        enableReinitialize
        validateOnChange
        validateOnBlur
      >
        {(formik: any) => {
          return (
            <Form>
              <InputGroup inputs={2}>
                <InputContainer>
                  <Input
                    id="nome"
                    name="usuario.nome"
                    label={"bennie"}
                    placeholder={getPlaceholder("placeholder_responsavel")}
                    error={formik.errors.usuario?.nome && formik.touched.usuario?.nome}
                    icon={<UserIcon width={20} height={20}></UserIcon>}
                  />
                </InputContainer>
                <InputContainer>
                  <Input
                    id="email"
                    name="usuario.email"
                    label={t("loginEmail")}
                    placeholder="example@gmail.com"
                    validate={validateEmail}
                    error={formik.errors.usuario?.email && formik.touched.usuario?.email}
                    icon={<Mail width={20} height={20}></Mail>}
                  />
                </InputContainer>
              </InputGroup>

              <InputGroup inputs={2}>
                <InputContainer>
                  <Input
                    id="senha"
                    type="password"
                    name="senha"
                    label={t("password")}
                    placeholder="******"
                    error={formik.errors.senha && formik.touched.senha}
                    icon={<Password width={20} height={20}></Password>}
                    onBlur={(event: any) => {
                      if (passwordConfirmationValidation(formik.values.senha)) {
                        formik.setFieldTouched("senhaConfirmar", true, true);
                      }
                    }}
                  />
                </InputContainer>
                <InputContainer>
                  <Input
                    id="senhaConfirmar"
                    type="password"
                    name="senhaConfirmar"
                    label={t("repeatPassword")}
                    placeholder="******"
                    error={formik.errors.senhaConfirmar && formik.touched.senhaConfirmar}
                    icon={<Password width={20} height={20}></Password>}
                  />
                </InputContainer>
              </InputGroup>

              <InputGroup inputs={3}>
                <InputContainer>
                  <Input
                    id="celular"
                    name="usuario.celular"
                    placeholder={t("cellPhonePlaceholder")}
                    label={t("cellPhone")}
                    error={formik.errors.usuario?.celular && formik.touched.usuario?.celular}
                    onInput={(event: any) => {
                      event.nativeEvent.target.value = masks.brCellphone(event.nativeEvent.target.value);
                    }}
                    icon={<Cellphone width={20} height={20}></Cellphone>}
                  />
                </InputContainer>
                <InputContainer>
                  <Input
                    id="cpfCnpj"
                    name="pessoa.cpfCnpj"
                    label={t("cpfCnpj")}
                    placeholder={t("cpfCnpj")}
                    onInput={(event: any) => {
                      event.nativeEvent.target.value = masks.cpfCnpj(event.nativeEvent.target.value);
                    }}
                    error={formik.errors.pessoa?.cpfCnpj && formik.touched.pessoa?.cpfCnpj}
                    icon={<HiDocumentText size={22} color={theme.primaryBackground} />}
                  />
                </InputContainer>
                <InputContainer>
                  <Input
                    id="emailUsuario"
                    name="pessoa.email"
                    label={t("contactEmail")}
                    placeholder="example@gmail.com"
                    validate={validateEmailUser}
                    error={formik.errors.pessoa?.email && formik.touched.pessoa?.email}
                    icon={<Mail width={20} height={20}></Mail>}
                  />
                </InputContainer>
              </InputGroup>

              <InputGroup inputs={1}>
                <Input
                  id="descricaoComercial"
                  name="estabelecimento.descricaoComercial"
                  label={t("details")}
                  placeholder={t("details")}
                  as="textarea"
                  error={formik.errors.estabelecimento?.descricaoComercial && formik.touched.estabelecimento?.descricaoComercial}
                  icon={<CgDetailsMore size={22} color={theme.primaryBackground} />}
                />
              </InputGroup>

              <Container>
                <ContainerText>
                  <Container>
                    <Text fontSize={16}>{t("paymentInfo")}</Text>
                  </Container>

                  <Container>
                    <Text fontSize={12}>{t("warning")}</Text>
                  </Container>
                </ContainerText>
              </Container>

              <InputGroup inputs={1}>
                <InputContainer>
                  <Input
                    as="select"
                    label={t("bankCode")}
                    id="codigoBanco"
                    name="pessoa.contaBancaria.codigoBanco"
                    placeholder={t("placeholder_banco")}
                    options={banks}
                    icon={<AiFillBank size={22} color={theme.primaryBackground} />}
                  />
                </InputContainer>
              </InputGroup>

              <InputGroup inputs={2}>
                <InputContainer>
                  <Input
                    id="nomeRazaoSocial"
                    name="pessoa.contaBancaria.nomeRazaoSocial"
                    label={t("companyName")}
                    placeholder={t("placeholder_responsavel")}
                    error={
                      formik.errors.pessoa?.contaBancaria?.nomeRazaoSocial &&
                      formik.touched.pessoa?.contaBancaria?.nomeRazaoSocial
                    }
                    icon={<UserIcon width={20} height={20}></UserIcon>}
                  />
                </InputContainer>
                <InputContainer>
                  <Input
                    id="cpfCnpjResponsavel"
                    name="pessoa.contaBancaria.cpfCnpjResponsavel"
                    label={t("ownerCpfCnpj")}
                    placeholder={t("cpfCnpj")}
                    onInput={(event: any) => {
                      event.nativeEvent.target.value = masks.cpfCnpj(event.nativeEvent.target.value);
                    }}
                    error={
                      formik.errors.pessoa?.contaBancaria?.cpfCnpjResponsavel &&
                      formik.touched.pessoa?.contaBancaria?.cpfCnpjResponsavel
                    }
                    icon={<HiDocumentText size={22} color={theme.primaryBackground} />}
                  />
                </InputContainer>
              </InputGroup>

              <InputGroup inputs={3}>
                <InputContainer>
                  <Input
                    id="contaBancaria"
                    name="pessoa.contaBancaria.contaBancaria"
                    type="text"
                    maxLength="13"
                    label={t("bankAccount")}
                    placeholder={t("placeholder_conta")}
                    onInput={(event: any) => {
                      event.nativeEvent.target.value = event.nativeEvent.target.value?.replace(/\D/g, "");
                    }}
                    error={
                      formik.errors.pessoa?.contaBancaria?.contaBancaria &&
                      formik.touched.pessoa?.contaBancaria?.contaBancaria
                    }
                    icon={<AiFillBank size={22} color={theme.primaryBackground} />}
                  />
                </InputContainer>
                <InputContainer>
                  <Input
                    id="digitoContaBancaria"
                    name="pessoa.contaBancaria.digitoContaBancaria"
                    maxLength="2"
                    label={t("bankAccountDigit")}
                    placeholder={t("placeholder_digito")}
                    onInput={(event: any) => {
                      event.nativeEvent.target.value = event.nativeEvent.target.value
                        ?.replace(/(?!\w|\s)./g, "")
                        .replace(/\s+/g, "");
                    }}
                    error={
                      formik.errors.pessoa?.contaBancaria?.digitoContaBancaria &&
                      formik.touched.pessoa?.contaBancaria?.digitoContaBancaria
                    }
                    icon={<AiFillBank size={22} color={theme.primaryBackground} />}
                  />
                </InputContainer>
                <InputContainer>
                  <Input
                    id="agenciaBancaria"
                    name="pessoa.contaBancaria.agenciaBancaria"
                    type="text"
                    maxLength="5"
                    label={t("agency")}
                    placeholder={t("placeholder_agencia")}
                    onInput={(event: any) => {
                      event.nativeEvent.target.value = event.nativeEvent.target.value?.replace(/\D/g, "");
                    }}
                    error={
                      formik.errors.pessoa?.contaBancaria?.agenciaBancaria &&
                      formik.touched.pessoa?.contaBancaria?.agenciaBancaria
                    }
                    icon={<AiFillBank size={22} color={theme.primaryBackground} />}
                  />
                </InputContainer>
              </InputGroup>

              <InputGroup inputs={1}>
                <InputContainer>
                  <Input
                    as="select"
                    label={t("transferInterval")}
                    id="intervaloDeTransferencia"
                    name="pessoa.contaBancaria.intervaloDeTransferencia"
                    error={
                      formik.errors.pessoa?.contaBancaria?.intervaloDeTransferencia &&
                      formik.touched.pessoa?.contaBancaria?.intervaloDeTransferencia
                    }
                    options={dropdownOptions}
                    icon={<BiTransfer size={22} color={theme.primaryBackground} />}
                  />
                </InputContainer>
              </InputGroup>

              <ButtonContainer>
                <Button type="submit" text={t("submit")} disabled={formik.isSubmitting || !formik.isValid} />
              </ButtonContainer>
            </Form>
          );
        }}
      </Formik>
    </Wrapper>
  );
}

export default SupplierProfile;

export const getServerSideProps: GetServerSideProps = async (ctx) => {
  const { res, resolvedUrl } = ctx;
  const token = parseCookies(ctx).access_token;
  const savedUser = parseCookies(ctx).user;

  if (!token) {
    return Util.sendToHomePage(res, undefined);
  }

  let language = parseCookies(ctx).language || "pt";
  if (ctx.locale != language && language) {
    return Util.redirectToSelectedLanguage(ctx, resolvedUrl);
  }

  if (savedUser) {
    let user = JSON.parse(savedUser) as User;
    if (user.perfilAtual == "C") {
      return Util.sendToAnyPage(res, "/profiles/user");
    }
  }

  return {
    props: {
      ...(await serverSideTranslations(ctx.locale as string, ["common"])),
    },
  };
};
