import { Wrapper, Title, SearchContainer, InputWrapper, ButtonContainer } from "./styles";
import { Util } from "../../../../core/utils/util";
import Head from "next/head";
import { parseCookies } from "nookies";
import { GetServerSideProps } from "next";
import { serverSideTranslations } from "next-i18next/serverSideTranslations";
import DataTableCustom from "../../../../app/components/DataTable";
import useClient from "../../../../core/hooks/useClient";
import { useCallback, useEffect, useState } from "react";
import ReactPaginate from "react-paginate";
import { FaSearch } from "react-icons/fa";
import { InputText } from "primereact/inputtext";
import Button from "../../../../app/components/Button";
import { useTheme } from "styled-components";
import { useTranslation } from "react-i18next";
import { Pagination } from "../../../../core/models/interfaces/Pagination";

function Client() {
  const { getDataListUserFilter } = useClient();
  //Tipagem Correta pra paginação.
  const [dataContent, setDataContent] = useState<Pagination>({} as Pagination);
  const [inputSearch, setInputSearch] = useState("");
  const theme = useTheme();
  const { t } = useTranslation("common");
  const columnsDataTable = [
    { field: "idCliente", header: "Código", type: "string" },
    { field: "nome", header: "Nome", type: "string" },
    { field: "cidade", header: "Cidade", type: "string" },
    { field: "estado", header: "Estado", type: "string" },
    { field: "telefone", header: "telefone", type: "string" },
    { field: "cpfCnpj", header: "CPF/CNPJ", type: "string" },
    { field: "dataNascimento", header: "Data Nascimento", type: "date" },
    { field: "urlFoto", header: "Perfil", type: "image" },
    {
      field: "buttonAction",
      header: "",
      type: "buttonAction",
      typeButton: [
        {
          page: "clients",
          label: "Detalhes",
        },
      ],
    },
  ];
  const [filterField, setFilterField] = useState({
    cliente: {
      buscaLivre: inputSearch,
    },
  });
  const [queryPage, setQueryPage] = useState({ page: 1, size: 6 });

  const getDataListFilterDto = useCallback(async (filter, query) => {
    const response = await getDataListUserFilter(filter, query);
    setDataContent(response.data);
  }, []);

  const handleKeyDown = useCallback(async (event: any) => {
    if (event.key === "Enter") {
      const response = await getDataListUserFilter({ cliente: { buscaLivre: event.target.value } }, queryPage);
      setDataContent(response.data);
    }
  }, []);

  useEffect(() => {
    getDataListFilterDto(filterField, queryPage);
  }, [getDataListFilterDto]);

  return (
    <Wrapper>
      <Head>
        <title>Relty - Clientes</title>
      </Head>
      <Title>Clientes</Title>

      <SearchContainer>
        <InputWrapper style={{ margin: "auto" }}>
          <span className="p-input-icon-left">
            <FaSearch size={19} color={theme.pageForeground} className="pi pi-search" />
            {/* @ts-ignore */}
            <InputText
              placeholder="Digite sua busca"
              className="p-inputtext-md"
              onKeyDown={(event) => handleKeyDown(event)}
              onChange={(i) => setInputSearch(i.target.value)}
            />
          </span>
          <ButtonContainer>
            <Button
              text="Buscar"
              onClick={() => getDataListFilterDto({ cliente: { buscaLivre: inputSearch } }, queryPage)}
            ></Button>
          </ButtonContainer>
        </InputWrapper>
      </SearchContainer>
      <br />

      {/* @ts-ignore */}
      <DataTableCustom values={dataContent?.content} columns={columnsDataTable} />
      <ReactPaginate
        forcePage={queryPage.page - 1}
        data-content="My Content"
        containerClassName={"Pagination"}
        marginPagesDisplayed={0}
        pageRangeDisplayed={4}
        pageCount={dataContent?.totalPages ? dataContent?.totalPages : 0}
        previousLabel={"<"}
        nextLabel={">"}
        onPageChange={(page) => getDataListFilterDto(filterField, { page: page.selected, size: queryPage.size })}
      />
    </Wrapper>
  );
}

export default Client;

export const getServerSideProps: GetServerSideProps = async (ctx) => {
  const { resolvedUrl } = ctx;

  let language = parseCookies(ctx).language || "pt";
  if (ctx.locale != language && language) {
    return Util.redirectToSelectedLanguage(ctx, resolvedUrl);
  }

  return {
    props: {
      ...(await serverSideTranslations(ctx.locale as string, ["common"])),
    },
  };
};
