import { serverSideTranslations } from 'next-i18next/serverSideTranslations';
import Head from 'next/dist/shared/lib/head';
import { SearchContainer, CardsGrid, CardContainer, NoData } from './styles';
import CardAdventure from '../../../../app/components/CardAdventure';
import HomeSearchBar from '../../../../app/components/HomeSearchBar';
import { AdventureListDto } from '../../../../core/models/interfaces/Adventure/AdventureListDto';
import { AdventureType } from '../../../../core/models/enums/AdventureType';
import useMasks from '../../../../core/hooks/useMasks';
import ReactPaginate from 'react-paginate';
import { useState } from 'react';
import { useTranslation } from 'next-i18next';
import { Util } from '../../../../core/utils/util';
import Router from 'next/router';
import { AdventureOrder } from '../../../../core/models/enums/AdventureOrder';
import { getApi } from '../../../../core/services/apiServerSide';
import { parseCookies } from 'nookies';
import { User } from '../../../../core/models/interfaces/User/User';

interface AdventuresProps {
  adventures?: AdventureListDto[];
  totalPages?: number;
  adventureType?: AdventureType | null;
  search?: string | string[] | null;
  page: number;
  price?: string | string[] | null;
  order?: string | null;
}

interface AdventureSearchParameters {
  search?: string | string[] | null;
  activeElement?: AdventureType | null;
  page: number;
  price?: string | string[] | null;
  order?: string | null;
}

const Adventure = (props: AdventuresProps) => {
  const { adventures, totalPages } = props;
  const { money, unMaskedMoney } = useMasks();
  const { t } = useTranslation('common');
  const [parameters, setParameters] = useState<AdventureSearchParameters>({
    page: props.page,
    search: props.search,
    activeElement: props.adventureType,
    price: props.price
      ? money(parseFloat(String(props.price)).toFixed(2)).maskedValue
      : props.price,
    order: props.order,
  });

  const handleSearchAventure = (
    event?: React.KeyboardEvent<HTMLInputElement>,
    page?: number
  ) => {
    if (event) {
      if (event.key != 'Enter') {
        return;
      } else {
        setParameters({ ...parameters, page: 1 });
      }
    }

    const { search, activeElement, price, order } = parameters;

    let finalUrl = Util.mountAdventureFilterParams(
      search as string,
      activeElement,
      String(unMaskedMoney(price as string)),
      order as AdventureOrder
    );
    let url = `/profiles/supplier/adventures?page=${page || 1}` + finalUrl;

    Router.replace(url);
  };

  const clearFilters = () => {
    parameters.order = '';
    parameters.activeElement = null;
    parameters.search = '';
    parameters.price = null;
    parameters.page = 1;

    handleSearchAventure();
  };

  const hrefBuilder = (page?: number): string => {
    const { search, activeElement, price, order } = parameters;
    return (
      `/profiles/supplier/adventures?page=${page || 1}` +
      Util.mountAdventureFilterParams(
        search as string,
        activeElement,
        String(unMaskedMoney(price as string)),
        order as AdventureOrder
      )
    );
  };

  const setPrice = (text?: string) => {
    setParameters({ ...parameters, price: text });
  };

  const setOrder = (text?: string) => {
    setParameters({ ...parameters, order: text });
    parameters.order = text;
    handleSearchAventure(undefined, 1);
  };

  const setSearch = (text?: string | string[] | null) => {
    setParameters({ ...parameters, search: text });
  };

  const setActiveElement = (newElement: AdventureType | null | undefined) => {
    setParameters({ ...parameters, activeElement: newElement });
    parameters.activeElement = newElement;
    handleSearchAventure();
  };

  const filters = {
    price: {
      value: parameters.price as string,
      visible: true,
      setMethod: setPrice,
    },
    order: {
      value: parameters.order as AdventureOrder,
      visible: true,
      setMethod: setOrder,
    },
  };

  return (
    <>
      <Head>
        <title>Relty - Minhas aventuras</title>
        <meta name="description" content="Relty ativades físicas e aventuras" />
        <link rel="icon" href="/assets/icon.png" />
      </Head>
      <SearchContainer>
        {/* ------- Search */}
        <HomeSearchBar
            activeElement={parameters.activeElement}
            setActiveElement={setActiveElement}
            search={parameters.search || null}
            setSearch={setSearch}
            handleSearchAventure={handleSearchAventure}
            advancedFilters={filters}
            clearFilters={clearFilters}
        />
        {/* ------- Search */}
      </SearchContainer>

      {adventures && adventures.length > 0 ? (
        <>
          <CardsGrid>
            {adventures?.map((item) => (
              <CardContainer key={item.id}>
                <CardAdventure adventure={item} isEditCrew />
              </CardContainer>
            ))}
          </CardsGrid>

          <ReactPaginate
            forcePage={parameters.page - 1}
            data-content="My Content"
            containerClassName={'Pagination'}
            pageCount={totalPages ? totalPages : 0}
            marginPagesDisplayed={0}
            pageRangeDisplayed={3}
            previousLabel={'<'}
            nextLabel={'>'}
            hrefBuilder={(page) => hrefBuilder(page)}
            onPageChange={(page) =>
              handleSearchAventure(undefined, page.selected + 1)
            }
          />
        </>
        ) : (
            <NoData>
              <h2>{t("noResults")}</h2>
            </NoData>
        )}
        </>
    )
};

export default Adventure;

export const getServerSideProps = async (ctx: any) => {
  const api = getApi(ctx);
  const { query, res, resolvedUrl, userSupplier } = ctx;
  const { page: _page, adventureType, search, price: _price, order } = query;
  const page = Number(_page);
  const price = Number(_price);
  const language = parseCookies(ctx).language || 'pt';
  const savedUser: User = JSON.parse(parseCookies(ctx).user);

  if (ctx.locale != language && language) {
    return Util.redirectToSelectedLanguage(ctx, resolvedUrl);
  }

  if (adventureType && Util.getValidAdventureType(adventureType) === null) {
    return Util.sendToSupplierAdventures(res, page);
  }

  if (price && (isNaN(price) || price < 0)) {
    return Util.sendToSupplierAdventures(res, page);
  }

  if (isNaN(page) || page < 1) {
    return Util.sendToSupplierAdventures(res);
  }

  console.log(savedUser);

  let body = {
    tipoAventura: Util.getValidAdventureType(adventureType),
    buscaLivre: search ? search : null,
    filtrarSomenteDisponivel: true,
    preco: price ? price?.toString() : null,
    estabelecimentoId: savedUser.estabelecimentoId,
  };

  const adventures = await api.post(
    `/WS/Aventura/Filtrar?page=${page - 1}&size=12 ${Util.mountAdventureOrder(
      String(order) as AdventureOrder
    )}`,
    body
  );

  return {
    props: {
      ...(await serverSideTranslations(ctx.locale as string, ['common'])),
      adventures: adventures.data.content,
      totalPages: adventures.data.totalPages,
      adventureType: Util.getValidAdventureType(adventureType),
      search: search || null,
      page: page,
      order: (order as string) || null,
      price: price ? price?.toString() : null,
    },
  };
};
