import axios from 'axios';
import { Form, Formik } from 'formik';
import { GetServerSideProps } from 'next';
import { useTranslation } from 'next-i18next';
import { serverSideTranslations } from 'next-i18next/serverSideTranslations';
import Head from 'next/head';
import { parseCookies } from 'nookies';
import { Button as ButtonPrimeReact } from 'primereact/button';
import { FileUpload } from 'primereact/fileupload';
import { InputSwitch } from 'primereact/inputswitch';
import { ListBox } from 'primereact/listbox';
import { Tag } from 'primereact/tag';
import { Toast } from 'primereact/toast';
import React, { useCallback, useEffect, useRef, useState } from 'react';
import { CgDetailsMore } from 'react-icons/cg';
import {
  IoIosAnalytics,
  IoIosRemove,
  IoLogoUsd,
  IoMdAlarm,
  IoMdAlbums,
  IoMdHome,
  IoMdTime,
  IoMdTrendingUp,
  IoMdWalk,
} from 'react-icons/io';
import { useTheme } from 'styled-components';
import * as Yup from 'yup';
import Button from '../../../../../app/components/Button';
import Input from '../../../../../app/components/CustomInput';
import FormLoading from '../../../../../app/components/FormLoading';
import ModalDefineTimeNewHours from '../../../../../app/components/ModalDefinedTime/ModalDefineTimeNewHours';
import { useGeolocationContext } from '../../../../../core/contexts/GeolocationContext';
import useAdventure from '../../../../../core/hooks/useAdventure';
import useEstablishment from '../../../../../core/hooks/useEstablishment';
import useMap from '../../../../../core/hooks/useMap';
import useMasks from '../../../../../core/hooks/useMasks';
import { Image } from '../../../../../core/models/interfaces/Image/Image';
import { masks } from '../../../../../core/utils/masks';
import { Util } from '../../../../../core/utils/util';
import {
  ContainerText,
  InputContainer,
  InputGroup,
  MapContainer,
  Text,
  Title,
  Wrapper,
} from './styles';

export default function NewAdventure() {
  const { t } = useTranslation('common');
  const { mapRef, loadMap, searchPlaces } = useMap();
  const { lastLocation, localizationMaps } = useGeolocationContext();
  const toastRef = useRef<any>();
  const [adventureImages, setAdventureImages] = useState<Image[]>([]);
  const [loading, setLoading] = useState(false);
  const theme = useTheme();
  const [selectedInclude, setSelectedInlude] = useState([]);
  const [selectedNotForget, setSelectedNotForget] = useState([]);
  const [visibleDialogDefinedTimeNewHours, setVisibleDialogDefinedTimeNewHours] = useState(false);
  const [checked, setChecked] = useState(false);
  const [checkedDuracao, setCheckedDuracao] = useState(false);
  const { getDataCategoria, getDataMaterial } = useEstablishment();
  const [listType, setListType] = useState([]);
  const { money } = useMasks();
  const [materialListInclude, setMaterialListInclude] = useState([]);
  const [materialListNotForget, setMaterialListNotForget] = useState([]);
  const [includeMaterialImutable, setIncludeMaterialImutable] = useState([]);
  const [totalSize, setTotalSize] = useState(0);
  const { createAdventure } = useAdventure();
  const dropdownOptions = [
    { key: t('easy'), value: 'F' },
    { key: t('medium'), value: 'M' },
    { key: t('hard'), value: 'D' },
  ];
  const parcelaOptions = [
    { key: '1x', value: 1 },
    { key: '2x', value: 2 },
    { key: '3x', value: 3 },
    { key: '4x', value: 4 },
    { key: '5x', value: 5 },
    { key: '6x', value: 6 },
    { key: '7x', value: 7 },
    { key: '8x', value: 8 },
    { key: '9x', value: 9 },
    { key: '10x', value: 10 },
    { key: '11x', value: 11 },
    { key: '12x', value: 12 },
  ];

  const listEstados = [
    { key: 'São Paulo, SP', value: 'São Paulo, SP' },
    { key: 'Paraná, PR', value: 'Paraná, PR' },
    { key: 'Santa Catarina, SC', value: 'Santa Catarina, SC' },
    { key: 'Rio Grande do Sul, RS', value: 'Rio Grande do Sul, RS' },
    { key: 'Mato Grosso do Sul, MS', value: 'Mato Grosso do Sul, MS' },
    { key: 'Rondônia, RO', value: 'Rondônia, RO' },
    { key: 'Acre, AC', value: 'Acre, AC' },
    { key: 'Amazonas, AM', value: 'Amazonas, AM' },
    { key: 'Roraima, RR', value: 'Roraima, RR' },
    { key: 'Pará, PA', value: 'Pará, PA' },
    { key: 'Amapá, AP', value: 'Amapá, AP' },
    { key: 'Tocantins, TO', value: 'Tocantins, TO' },
    { key: 'Maranhão, MA', value: 'Maranhão, MA' },
    { key: 'Rio Grande do Norte, RN', value: 'Rio Grande do Norte, RN' },
    { key: 'Paraíba, PB', value: 'Paraíba, PB' },
    { key: 'Pernambuco, PE', value: 'Pernambuco, PE' },
    { key: 'Alagoas, AL', value: 'Alagoas, AL' },
    { key: 'Sergipe, SE', value: 'Sergipe, SE' },
    { key: 'Bahia, BA', value: 'Bahia, BA' },
    { key: 'Minas Gerais, MG', value: 'Minas Gerais, MG' },
    { key: 'Rio de Janeiro, RJ', value: 'Rio de Janeiro, RJ' },
    { key: 'Mato Grosso, MT', value: 'Mato Grosso, MT' },
    { key: 'Goiás, GO', value: 'Goiás, GO' },
    { key: 'Distrito Federal, DF', value: 'Distrito Federal, DF' },
    { key: 'Piauí, PI', value: 'Piauí, PI' },
    { key: 'Ceará, CE', value: 'Ceará, CE' },
    { key: 'Espírito Santo, ES', value: 'Espírito Santo, ES' }
  ]

  const InputSwitchCustomMetrosQuilometros = (props: any) => {
    return (
      <>
        <div
          style={{
            display: 'flex',
            flexDirection: 'row',
            alignItems: 'center',
          }}
        >
          <p>{props.label}</p> &nbsp; &nbsp; &nbsp;{' '}
          <InputSwitch
            checked={checked}
            onChange={(e: any) => setChecked((oldValue) => !oldValue)}
          />
        </div>
      </>
    );
  };

  const InputSwitchCustomDiasHoras = (props: any) => {
    return (
      <>
        <div
          style={{
            display: 'flex',
            flexDirection: 'row',
            alignItems: 'center',
          }}
        >
          <p>{props.label}</p> &nbsp; &nbsp; &nbsp;{' '}
          <InputSwitch
            checked={checkedDuracao}
            onChange={(e: any) => setCheckedDuracao((oldValue) => !oldValue)}
          />
        </div>
      </>
    );
  };

  const initialMap = () => {
    loadMap(lastLocation, () => {});
  };

  const getCategoria = useCallback(async () => {
    const response = await getDataCategoria();
    setListType(
      response.data.map((category: any) => {
        return {
          ...category,
          key: category.nome,
          value: category.id,
        };
      })
    );
  }, [setListType]);

  const getMaterial = useCallback(async () => {
    const response = await getDataMaterial();
    setMaterialListInclude(response.data);
    setMaterialListNotForget(response.data);
    setIncludeMaterialImutable(response.data);
  }, []);

  useEffect(() => {
    initialMap();
    getCategoria();
    getMaterial();
  }, []);

  useEffect(() => {
    filterListMaterialNotForget(selectedInclude);
    filterListMaterialInclude(selectedNotForget);
  }, [selectedNotForget, selectedInclude]);

  const filterListMaterialNotForget = (selectedInclude: any) => {
    if (selectedInclude) {
      let listId = selectedInclude.map((i: any) => i.id);
      let filterItem = includeMaterialImutable.filter(
        // @ts-ignore 
        (y) => !listId.includes(y.id)
      );
      setMaterialListNotForget(filterItem);
    }
  };

  const filterListMaterialInclude = (selectedInclude: any) => {
    if (selectedInclude) {
      let listId = selectedInclude.map((i: any) => i.id);
      let filterItem = includeMaterialImutable.filter(
        // @ts-ignore
        (y) => !listId.includes(y.id)
      );
      setMaterialListInclude(filterItem);
    }
  };

  const onTemplateRemove = (file: any, callback: any) => {
    let filteredImages: Image[] = adventureImages.filter(
      (image: Image) => image.nome != file.name
    );
    setAdventureImages(filteredImages);
    setTotalSize(totalSize - file.size);
    callback();
  };

  const onSubmit = async (event: any) => {
    console.log('event ===>', event)
    // @ts-ignore
    event.categoria = event.categoria ? event.categoria : listType[0].value;
    event.nivel = event.nivel ? event.nivel : dropdownOptions[0].value;
    event.parcelas = event.parcelas ? event.parcelas : parcelaOptions[0].value;

    let horarios = window.sessionStorage.getItem('horarios');
    horarios = horarios ? JSON.parse(horarios as string) : [];

    // if (!localizationMaps)
    //   return Util.showErrorToast(
    //     toastRef,
    //     'Erro: Selecione um local no mapa antes de prosseguir!'
    // );
    if (adventureImages.length == 0)
      return Util.showErrorToast(
        toastRef,
        'Erro: Selecione ao menos 1 imagem para sua aventura!'
    );
    if (!horarios || horarios.length == 0)
      return Util.showErrorToast(
        toastRef,
        'Erro: É necessário definir ao menos um horário para a Aventura!'
    );

    const body = {
      categoria: listType.find((item: any) => item.id == event.categoria),
      descricao: event.detalhes,
      duracao: event.duracao,
      endereco: {
        bairro: event.endereco.bairro,
        cep: event.endereco.cep,
        cidade: event.local,
        complemento: event.endereco.complemento,
        latitude: localizationMaps?.lat,
        longitude: localizationMaps?.lng,
        numero: event.endereco.numero,
        rua: event.endereco.rua,
        uf: event.endereco.uf,
      },
      horarios: horarios,
      horasCompraAntecedencia: event.horasCompraAntecedencia,
      imagens: adventureImages.map((img) => {
        return {
          bytes: img.bytes,
          imagem: img,
        };
      }),
      local: event.local,
      materiais: selectedNotForget.map((material: any) => {
          return { material: material, tipoMaterial: 'O' };
        })
        .concat(
          selectedInclude.map((material: any) => {
            return { material: material, tipoMaterial: 'I' };
          })
        ),
      nivel: event.nivel,
      percurso: event.percurso,
      preco: event.preco.replace('R$', '').replace('.', '').replace(',', '.'),
      quantidadeParcela: event.parcelas,
      tipoDuracao: checkedDuracao ? 'HORAS' : 'DIAS',
      tipoPercurso: checked ? 'QUILOMETROS' : 'METROS',
      titulo: event.titulo,
    };

    try {
      await createAdventure(body);
      Util.showSucessToast(toastRef, 'Aventura criada com sucesso!');
      window.location.href = '/profiles/supplier/adventures';
    } catch (e) {
      Util.showErrorToast(toastRef, 'Erro ao criar aventura!');
    }
  };

  const actualizeCep = async (cep: string, formik: any) => {
    axios.get('https://viacep.com.br/ws/' + cep + '/json/').then((response) => {
      if (!response.data) return console.log('Erro, ');
      if (response.data.error) return console.log('Cep inválido');
      let { logradouro, bairro, uf } = response.data;
      formik.setFieldValue('endereco.rua', logradouro);
      formik.setFieldValue('endereco.bairro', bairro);
      formik.setFieldValue('endereco.uf', uf);
    });
  };

  const chooseOptions = {
    icon: 'pi pi-fw pi-images',
    iconOnly: true,
    className: 'custom-choose-btn p-button-rounded p-button-outlined',
  };
  const uploadOptions = {
    icon: 'pi pi-fw pi-cloud-upload',
    iconOnly: true,
    style: { display: 'none' },
    className:
      'custom-upload-btn p-button-success p-button-rounded p-button-outlined',
  };
  const cancelOptions = {
    icon: 'pi pi-fw pi-times',
    iconOnly: true,
    className:
      'custom-cancel-btn p-button-danger p-button-rounded p-button-outlined',
  };

  const invoiceUploadHandler = ({files}: any) => {
    console.log('invoiceUploadHandler ==>', files)
    const [file] = files;

    const fileReader = new FileReader();
    fileReader.onload = (e) => {
      let result: string = e?.target?.result as string;
      let image: Image = {
        bytes: result.split(',')[1],
        nome: file.name,
        contentType: result.split(';')[0].split(':')[1],
      };
      setAdventureImages(adventureImages.concat(image));
    };
    fileReader.readAsDataURL(file);
  };

  const onTemplateClear = () => {
    setAdventureImages([]);
  };

  const itemTemplate = (file: any, props: any) => {
    return (
      <div className="p-d-flex p-ai-center p-flex-wrap">
        <div className="p-d-flex p-ai-center" style={{ width: '40%' }}>
          <img alt={file.name} role="presentation" src={file.objectURL} width={100} />
          <span className="p-d-flex p-dir-col p-text-left p-ml-3">
            {file.name}
            <small>{new Date().toLocaleDateString()}</small>
          </span>
        </div>
        {/* @ts-ignore */}
        <Tag value={props.formatSize} severity="warning" />
        {/* @ts-ignore */}
        <ButtonPrimeReact
          icon="pi pi-times"
          className="p-button-outlined p-button-rounded p-button-danger p-ml-auto"
          onClick={() => onTemplateRemove(file, props.onRemove)}
        />
      </div>
    );
  };

  const emptyTemplate = () => {
    return (
      <div className="p-d-flex p-ai-center p-dir-col">
        <i
          className="pi pi-image p-mt-3 p-p-5"
          style={{
            fontSize: '5em',
            borderRadius: '50%',
            backgroundColor: 'var(--surface-b)',
            color: 'var(--surface-d)',
          }}
        ></i>
        <span style={{ fontSize: '1.2em', color: 'var(--text-color-secondary)' }} className="p-my-5">
          Arraste e solte a imagem aqui
        </span>
      </div>
    );
  };

  const initialValues = {
    categoria: '',
    nivel: '',
    titulo: '',
    local: '',
    preco: '',
    horasCompraAntecedencia: '',
    parcelas: '',
    duracao: '',
    percurso: '',
    detalhes: '',
    endereco: {
      cep: '',
      rua: '',
      bairro: '',
      uf: '',
      numero: '',
      complemento: '',
    },
  };

  const validationSchema = Yup.object({
    titulo: Yup.string().required(t('requiredField')),
    local: Yup.string().required(t('requiredField')),
    preco: Yup.string()
      .required(t('requiredField'))
      .test(
        'É maior que R$0,00',
        'O valor deve ser maior que R$0,00',
        (value: any) => {
          return value != 'R$0,00';
        }
      ),
    horasCompraAntecedencia: Yup.number().required(t('requiredField')),
    duracao: Yup.number().required(t('requiredField')),
    percurso: Yup.number().required(t('requiredField')),
    detalhes: Yup.string().required(t('requiredField')),
    endereco: Yup.object().shape({
      cep: Yup.string().required(t('requiredField')),
      rua: Yup.string().required(t('requiredField')),
      bairro: Yup.string().required(t('requiredField')),
      uf: Yup.string().required(t('requiredField')),
      numero: Yup.number().required(t('requiredField')),
      complemento: Yup.string(),
    }),
  });

  return (
    <Wrapper>
      <Head>
        <title>Relty - Nova Aventura</title>
      </Head>
      <FormLoading show={loading} />
      {/* @ts-ignore */}
      <Toast ref={toastRef} position="top-center" />
      <Title> Nova Aventura </Title>

      <ModalDefineTimeNewHours visible={visibleDialogDefinedTimeNewHours} onHide={() => { setVisibleDialogDefinedTimeNewHours(!visibleDialogDefinedTimeNewHours); }}/>

      <Formik
        initialValues={initialValues}
        validationSchema={validationSchema}
        onSubmit={onSubmit}
        enableReinitialize
        validateOnChange
        validateOnBlur
      >
        {(formik: any) => {
          return (
            <Form>
              {/* @ts-ignore */}
              <FileUpload
                name="demo"
                itemTemplate={itemTemplate}
                emptyTemplate={emptyTemplate}
                chooseOptions={chooseOptions}
                uploadOptions={uploadOptions}
                accept="image/*"
                customUpload={true}
                multiple={true}
                onSelect={invoiceUploadHandler}
                onClear={onTemplateClear}
                cancelOptions={cancelOptions}
              />

              <ContainerText>
                <Text fontSize={12}> Recomendamos adicionar até 7 fotos </Text>
              </ContainerText>

              <ContainerText>
                <Text> Informações </Text>
              </ContainerText>

              <InputGroup inputs={2}>
                <InputContainer style={{ flexDirection: 'column' }}>
                  <Input
                    as="select"
                    options={listType}
                    name="categoria"
                    label="Tipo"
                    icon={
                      <IoMdAlbums size={16} color={theme.primaryBackground} />
                    }
                  />
                </InputContainer>
                <InputContainer>
                  <Input
                    as="select"
                    label="Nivel"
                    options={dropdownOptions}
                    name="nivel"
                    icon={
                      <IoMdTrendingUp
                        size={16}
                        color={theme.primaryBackground}
                      />
                    }
                  />
                </InputContainer>
              </InputGroup>

              <InputGroup inputs={3}>
                <InputContainer>
                  <Input
                    id="tituloAventura"
                    type="name"
                    name="titulo"
                    label="Título da aventura"
                    placeholder="Título da aventura"
                    icon={
                      <IoIosAnalytics
                        size={16}
                        color={theme.primaryBackground}
                      />
                    }
                  />
                </InputContainer>
                <InputContainer>
                  <Input id="municipioAventura" list="categoria" name="local" label="Município/Estado" placeholder="Município/Estado" icon={ <IoMdHome size={16} color={theme.primaryBackground} /> } />
                  <datalist id="categoria">
                    {listEstados.map((item: any, key: any) =>
                      <option key={key.key} value={item.key} />
                    )}
                  </datalist>
                </InputContainer>
                <InputContainer>
                  <Input
                    id="valorPessoa"
                    type="text"
                    name="preco"
                    label="Valor por pessoa(R$)"
                    placeholder="00,00"
                    icon={
                      <IoLogoUsd size={16} color={theme.primaryBackground} />
                    }
                    maxLength={10}
                    onInput={(event: any) => {
                      event.nativeEvent.target.value = money(
                        event.target.value
                      ).maskedValue;
                    }}
                  />
                </InputContainer>
              </InputGroup>

              <InputGroup inputs={2}>
                <InputContainer
                  title="É o tempo de antecedência mínima que o cliente pode
                                        comprar sua aventura. Ex: Colocando 24 horas o clinete tem que comprar
                                        a aventura pelo menos um dia antes de determinado horário"
                >
                  <Input
                    label="Horas de antecedencia na compra"
                    name="horasCompraAntecedencia"
                    placeholder="24h"
                    maxLength={3}
                    onKeyPress={(event: any) => {
                      if (
                        event.nativeEvent.target.value.length == 0 &&
                        event.key == '0'
                      )
                        event.preventDefault();
                      else if (!/[0-9]/.test(event.key)) event.preventDefault();
                    }}
                    icon={
                      <IoMdAlarm size={16} color={theme.primaryBackground} />
                    }
                  />
                </InputContainer>
                <InputContainer>
                  <Input
                    as="select"
                    label="Quantidade de parcelas permitidas"
                    name="parcelas"
                    options={parcelaOptions}
                    icon={
                      <IoIosRemove size={16} color={theme.primaryBackground} />
                    }
                  />
                </InputContainer>
              </InputGroup>

              <InputGroup inputs={2}>
                <InputContainer>
                  <Input
                    id="duracao"
                    name="duracao"
                    label={
                      <InputSwitchCustomDiasHoras label="Duração dias/horas" />
                    }
                    placeholder={checkedDuracao ? 'Horas' : 'Dias'}
                    onKeyPress={(event: any) => {
                      if (
                        event.nativeEvent.target.value.length == 0 &&
                        event.key == '0'
                      )
                        event.preventDefault();
                      else if (!/[0-9]/.test(event.key)) event.preventDefault();
                    }}
                    icon={
                      <IoMdTime size={16} color={theme.primaryBackground} />
                    }
                  />
                </InputContainer>
                <InputContainer>
                  <Input
                    id="percuso"
                    name="percurso"
                    onKeyPress={(event: any) => {
                      if (
                        event.nativeEvent.target.value.length == 0 &&
                        event.key == '0'
                      )
                        event.preventDefault();
                      else if (!/[0-9]/.test(event.key)) event.preventDefault();
                    }}
                    label={ <InputSwitchCustomMetrosQuilometros label="Percurso metros/quilômetros" /> }
                    placeholder={checked ? 'Km' : 'Metros'}
                    icon={
                      <IoMdWalk size={16} color={theme.primaryBackground} />
                    }
                  />
                </InputContainer>
              </InputGroup>

              <ContainerText>
                <Text fontSize={16} style={{ margin: '0' }}>
                  Detalhes da aventura
                </Text>
              </ContainerText>

              <InputGroup inputs={1}>
                <Input
                  as="textarea"
                  name="detalhes"
                  label="Detalhes"
                  icon={ <CgDetailsMore size={16} color={theme.primaryBackground} /> }
                />
              </InputGroup>

              <ContainerText>
                <Text fontSize={16}>O que estará incluso</Text>
              </ContainerText>

              <InputGroup inputs={1}>
                {/* @ts-ignore */}
                <ListBox
                  value={selectedInclude}
                  virtualScrollerOptions={{
                    itemSize: 38,
                    scrollHeight: '200px',
                  }}
                  options={materialListInclude}
                  multiple
                  filter
                  optionLabel="descricao"
                  onChange={(e) => setSelectedInlude(e.target.value)}
                />
              </InputGroup>

              <ContainerText>
                <Text fontSize={16}>O que não pode esquecer</Text>
              </ContainerText>

              <InputGroup inputs={1}>
                {/* @ts-ignore */}
                <ListBox
                  value={selectedNotForget}
                  onChange={(e) => setSelectedNotForget(e.target.value)}
                  virtualScrollerOptions={{
                    itemSize: 38,
                    scrollHeight: '200px',
                  }}
                  options={materialListNotForget}
                  multiple
                  filter
                  optionLabel="descricao"
                />
              </InputGroup>

              <ContainerText>
                <Text fontSize={16}>Aonde vamos nos encontrar</Text>
              </ContainerText>

              <InputGroup inputs={3}>
                <InputContainer>
                  <Input
                    type="text"
                    name="endereco.cep"
                    label="CEP"
                    placeholder="CEP"
                    onInput={(event: any) => {
                      event.nativeEvent.target.value = masks.cep(
                        event.nativeEvent.target.value
                      );
                      let cep = event.nativeEvent.target.value;
                      if (cep.length === 9) actualizeCep(cep, formik);
                    }}
                    icon={
                      <IoMdHome size={16} color={theme.primaryBackground} />
                    }
                  />
                </InputContainer>
                <InputContainer>
                  <Input
                    type="text"
                    name="endereco.rua"
                    label="Rua"
                    placeholder="Rua"
                    icon={ <IoMdHome size={16} color={theme.primaryBackground} /> }
                  />
                </InputContainer>
                <InputContainer>
                  <Input
                    type="text"
                    name="endereco.bairro"
                    label="Bairro"
                    placeholder="Bairro"
                    icon={ <IoMdHome size={16} color={theme.primaryBackground} /> }
                  />
                </InputContainer>
              </InputGroup>

              <InputGroup inputs={3}>
                <InputContainer>
                  <Input
                    type="text"
                    name="endereco.uf"
                    label="UF"
                    placeholder="UF"
                    maxLength={2}
                    icon={ <IoMdHome size={16} color={theme.primaryBackground} /> }
                  />
                </InputContainer>
                <InputContainer>
                  <Input
                    type="text"
                    name="endereco.numero"
                    onKeyPress={(event: any) => {
                      if ( event.nativeEvent.target.value.length == 0 && event.key == '0') event.preventDefault();
                      else if (!/[0-9]/.test(event.key)) event.preventDefault();
                    }}
                    label="Número"
                    placeholder="Número"
                    icon={
                      <IoMdHome size={16} color={theme.primaryBackground} />
                    }
                  />
                </InputContainer>
                <InputContainer>
                  <Input
                    type="text"
                    name="endereco.complemento"
                    label="Complemento"
                    placeholder="Complemento"
                    icon={
                      <IoMdHome size={16} color={theme.primaryBackground} />
                    }
                  />
                </InputContainer>
              </InputGroup>

              <ContainerText>
                <Text fontSize={16}>
                  Confira o PIN no mapa, e ajuste selecionando no mapa
                </Text>
              </ContainerText>

              <InputGroup inputs={1}>
                <MapContainer ref={mapRef} />
              </InputGroup>

              <Button
                text="Definir Horário"
                type="button"
                onClick={() => setVisibleDialogDefinedTimeNewHours(true)}
              />
              <br />
              <Button type="submit" text="Salvar aventura" />
            </Form>
          );
        }}
      </Formik>
    </Wrapper>
  );
}

export const getServerSideProps: GetServerSideProps = async (ctx) => {
  const { res, resolvedUrl } = ctx;

  const token = parseCookies(ctx).access_token;
  if (!token) {
    return Util.sendToHomePage(res, undefined);
  }

  let language = parseCookies(ctx).language || 'pt';
  if (ctx.locale != language && language) {
    return Util.redirectToSelectedLanguage(ctx, resolvedUrl);
  }

  return {
    props: {
      ...(await serverSideTranslations(ctx.locale as string, ['common'])),
    },
  };
};
