import { Wrapper, Title, SearchContainer, InputWrapper, ButtonContainer } from "./styles";
import { Util } from "../../../../core/utils/util";
import Head from "next/head";
import { parseCookies } from "nookies";
import { GetServerSideProps } from "next";
import { serverSideTranslations } from "next-i18next/serverSideTranslations";
import DataTableCustom from "../../../../app/components/DataTable";
import useEstablishment from "../../../../core/hooks/useEstablishment";
import { useCallback, useEffect, useState } from "react";
import ReactPaginate from "react-paginate";
import { FaSearch } from "react-icons/fa";
import { InputText } from "primereact/inputtext";
import Button from "../../../../app/components/Button";
import { useTheme } from "styled-components";
import { useTranslation } from "react-i18next";
import { Pagination } from "../../../../core/models/interfaces/Pagination";

function AdminSupplier() {
  const { getDataListFilter } = useEstablishment();
  const [dataContent, setDataContent] = useState<Pagination>({} as Pagination);
  const [inputSearch, setInputSearch] = useState("");
  const theme = useTheme();
  const { t } = useTranslation("common");
  const columnsDataTable = [
    { field: "id", header: "Código", type: "string" },
    { field: "nome", header: "Nome", type: "string" },
    { field: "cpfCnpj", header: "CPF/CNPJ", type: "string" },
    { field: "telefone", header: "Telefone", type: "string" },
    { field: "descricaoComercial", header: "Descrição", type: "textLong" },
    { field: "email", header: "Email", type: "string" },
    { field: "urlFoto", header: "Perfil", type: "image" },
    {
      field: "buttonAction",
      header: "",
      type: "buttonAction",
      typeButton: [
        {
          page: "establishment",
          label: "Detalhes",
        },
        {
          label: "Alterar comissão",
          page: "establishment",
        },
      ],
    },
  ];
  const [filterField, setFilterField] = useState({
    nome: inputSearch,
  });
  const [queryPage, setQueryPage] = useState({ page: 1, size: 6 });

  const getDataListFilterDto = useCallback(async (filter, query) => {
    const response = await getDataListFilter(filter, query);
    setDataContent(response.data);
  }, []);

  const handleKeyDown = useCallback(async (event: any) => {
    if (event.key === "Enter") {
      const response = await getDataListFilter({ nome: event.target.value }, { page: 0, size: queryPage.size });
      setDataContent(response.data);
    }
  }, []);

  useEffect(() => {
    getDataListFilterDto(filterField, queryPage);
  }, [getDataListFilterDto]);

  return (
    <Wrapper>
      <Head>
        <title>Relty - Fornecedores</title>
      </Head>
      <Title>Fornecedores</Title>

      <SearchContainer>
        <InputWrapper style={{ margin: "auto" }}>
          <span className="p-input-icon-left">
            <FaSearch size={19} color={theme.pageForeground} className="pi pi-search" />
            {/* @ts-ignore */}
            <InputText
              placeholder="Digite sua busca"
              className="p-inputtext-md"
              onKeyDown={(event) => handleKeyDown(event)}
              onChange={(i) => setInputSearch(i.target.value)}
            />
          </span>
          <ButtonContainer>
            <Button
              text="Buscar"
              onClick={() => getDataListFilterDto({ nome: inputSearch }, { page: 0, size: queryPage.size })}
            ></Button>
          </ButtonContainer>
        </InputWrapper>
      </SearchContainer>
      <br />
      {/* @ts-ignore */}
      <DataTableCustom values={dataContent.content} columns={columnsDataTable} />
      <ReactPaginate
        forcePage={queryPage.page - 1}
        data-content="My Content"
        containerClassName={"Pagination"}
        marginPagesDisplayed={0}
        pageRangeDisplayed={4}
        pageCount={dataContent.totalPages ? dataContent.totalPages : 0}
        previousLabel={"<"}
        nextLabel={">"}
        onPageChange={(page) =>
          getDataListFilterDto({ nome: inputSearch }, { page: page.selected, size: queryPage.size })
        }
      />
    </Wrapper>
  );
}

export default AdminSupplier;

export const getServerSideProps: GetServerSideProps = async (ctx) => {
  const { resolvedUrl } = ctx;

  let language = parseCookies(ctx).language || "pt";
  if (ctx.locale != language && language) {
    return Util.redirectToSelectedLanguage(ctx, resolvedUrl);
  }

  return {
    props: {
      ...(await serverSideTranslations(ctx.locale as string, ["common"])),
    },
  };
};
