import { Wrapper, Title, SearchContainer, InputWrapper, ButtonContainer } from "./styles";
import { Util } from "../../../../core/utils/util";
import Head from "next/head";
import { parseCookies } from "nookies";
import { GetServerSideProps } from "next";
import { serverSideTranslations } from "next-i18next/serverSideTranslations";
import DataTableCustom from '../../../../app/components/DataTable'
import React, { useCallback, useEffect, useState } from "react";
import ReactPaginate from "react-paginate";
import { FaSearch } from "react-icons/fa";
import { InputText } from "primereact/inputtext";
import Button from "../../../../app/components/Button";
import { useTheme } from "styled-components";
import useMaterial from "../../../../core/hooks/useMaterial";
import  MaterialModal from "../../../../app/components/MaterialModal";
import { getApi } from "../../../../core/services/apiServerSide";

export default function Materials(props: any) {

    const { listMaterials } = props;

    const { getDataMaterial } = useMaterial()
    const [dataContent, setDataContent] = useState<any>([])
    const [inputSearch, setInputSearch] = useState("")
    const theme = useTheme();
    const [visible, setVisible] = useState(false)
    const columnsDataTable = [
        { field: 'descricao', header: 'PT', type: 'string' },
        { field: 'descricaoEn', header: 'EN', type: 'string' },
        { field: 'descricaoEs', header: 'ES', type: 'string' },
        { field: 'buttonAction', header: '', type: 'buttonAction', typeButton: [
            {
                page: 'material',
                label: 'Editar material'
            },
            {
                page: 'material',
                label: 'Apagar material'
            }
        ]}
    ]
    const [filterField, setFilterField] = useState({
        descricao: inputSearch
    })
    const [queryPage, setQueryPage] = useState({ page: 1, size: 6 })


    const getDataListMaterialFilterDto = useCallback(async (filter, query) => {
        const response = await getDataMaterial(filter, query);
        setDataContent(response.data)
    }, []);

    const handleKeyDown = useCallback(async (event: any) => {
        if (event.key === 'Enter') {
            getDataListMaterialFilterDto({ descricao: event.target.value }, { page: 0, size: queryPage.size })
        }
    }, [])

    useEffect(() => {
        if(listMaterials){
            setDataContent(listMaterials)
        } else {
            getDataListMaterialFilterDto(filterField, queryPage)
        }
    }, [getDataListMaterialFilterDto])

    return (
        <Wrapper>
            <Head>
                <title>Relty - Materiais</title>
            </Head>
            <Title>Materiais</Title>
        
            <SearchContainer>
                <InputWrapper style={{ margin: 'auto' }}>
                <span className="p-input-icon-left">
                    <FaSearch size={19} color={theme.pageForeground} className="pi pi-search" />
                    {/* @ts-ignore */}
                    <InputText
                        placeholder="Digite sua busca"
                        className="p-inputtext-md"
                        onKeyDown={(event) => handleKeyDown(event)}
                        onChange={(i) => setInputSearch(i.target.value)}
                    />
                </span>
                <ButtonContainer>
                    <Button text="Buscar" onClick={() => getDataListMaterialFilterDto({ descricao: inputSearch }, { page: 0, size: queryPage.size })}></Button>
                </ButtonContainer>
                </InputWrapper>
            </SearchContainer>

            <SearchContainer>
                <InputWrapper style={{ justifyContent: 'start', padding: '0 0 12px' }}>
                <ButtonContainer style={{ width: 'inherit' }}>
                    <Button text="Listar materiais" style={{ margin: '10px' }} onClick={() => { getDataListMaterialFilterDto(filterField, { page: 0, size: 6 }), setInputSearch("") } }></Button>
                    <Button text="Novo material" style={{ margin: '10px' }} onClick={() => setVisible(true) }></Button>
                    <MaterialModal visible={visible} onHide={() => { setVisible(!visible) }}/>
                </ButtonContainer>
                </InputWrapper>
            </SearchContainer>
{/* @ts-ignore */}
            <DataTableCustom values={dataContent.content} columns={columnsDataTable} />
            <ReactPaginate
                forcePage={queryPage.page - 1}
                data-content="My Content"
                containerClassName={"Pagination"}
                marginPagesDisplayed={0}
                pageRangeDisplayed={4}
                pageCount={dataContent.totalPages ? dataContent.totalPages : 0}            
                previousLabel={"<"}
                nextLabel={">"}
                onPageChange={(page) => getDataListMaterialFilterDto({ descricao: inputSearch }, { page: page.selected, size: queryPage.size })}
            />
        </Wrapper>
    );
}

export const getServerSideProps: GetServerSideProps = async (ctx) => {
    const { resolvedUrl, query } = ctx;
    const api = getApi(ctx);
    let filtro = { descricao: query.search || "" }

    let language = parseCookies(ctx).language || "pt";
    if (ctx.locale != language && language) {
        return Util.redirectToSelectedLanguage(ctx, resolvedUrl);
    }

    const materials = await api.post(`/WS/Material/Listagem/Filtrar`, filtro, {
        params: {
            page: 0,
            size: 6,
            sortDirection: 'ASC',
            sortField: 'descricao'
        }
    })

    return {
        props: {
            ...(await serverSideTranslations(ctx.locale as string, ["common"])),
            listMaterials: materials.data
        },
    };
};