import { Wrapper, Title, SearchContainer, InputWrapper, ButtonContainer } from "./styles";
import { Util } from "../../../../core/utils/util";
import Head from "next/head";
import { parseCookies } from "nookies";
import { GetServerSideProps } from "next";
import { serverSideTranslations } from "next-i18next/serverSideTranslations";
import DataTableCustom from '../../../../app/components/DataTable'
import React, { useCallback, useEffect, useState } from "react";
import ReactPaginate from "react-paginate";
import { FaSearch } from "react-icons/fa";
import { InputText } from "primereact/inputtext";
import Button from "../../../../app/components/Button";
import { useTheme } from "styled-components";
import useCategory from "../../../../core/hooks/useCategory";
import CategoryModal from "../../../../app/components/CategoryModal";
import { getApi } from "../../../../core/services/apiServerSide";

export default function Categories(props: any) {

    const { listCategories } = props;

    const { getDataCategory } = useCategory()
    const [dataContent, setDataContent] = useState<any>([])
    const [inputSearch, setInputSearch] = useState("")
    const theme = useTheme();
    const [visible, setVisible] = useState(false)
    const columnsDataTable = [
        { field: 'nomePt', header: 'PT', type: 'string' },
        { field: 'nomeEn', header: 'EN', type: 'string' },
        { field: 'nomeEs', header: 'ES', type: 'string' },
        { field: 'tipoAventura', header: 'Elemento', type: 'elementIcon' },
        { field: 'buttonAction', header: '', type: 'buttonAction', typeButton: [
            {
                page: 'category',
                label: 'Editar categoria'
            },
            {
                page: 'category',
                label: 'Apagar categoria'
            }
        ]}
    ]
    const [filterField, setFilterField] = useState({
        nome: inputSearch
    })
    const [queryPage, setQueryPage] = useState({ page: 1, size: 6 })

    const getDataListCategoryFilterDto = useCallback(async (filter, query) => {
        const response = await getDataCategory(filter, query);
        setDataContent(response.data)
    }, []);

    const handleKeyDown = useCallback(async (event: any) => {
        if (event.key === 'Enter') {
            getDataListCategoryFilterDto({ nome: event.target.value }, { page: 0, size: queryPage.size })
        }
    }, [])

    useEffect(() => {
        if(listCategories){
            setDataContent(listCategories)
        } else {
            getDataListCategoryFilterDto(filterField, queryPage)
        }
    }, [getDataListCategoryFilterDto])

    return (
        <Wrapper>
            <Head>
                <title>Relty - Categorias</title>
            </Head>
            <Title>Categorias</Title>
        
            <SearchContainer>
                <InputWrapper style={{ margin: 'auto' }}>
                <span className="p-input-icon-left">
                    <FaSearch size={19} color={theme.pageForeground} className="pi pi-search" />
                    {/* @ts-ignore */}
                    <InputText
                        placeholder="Digite sua busca"
                        className="p-inputtext-md"
                        onKeyDown={(event) => handleKeyDown(event)}
                        onChange={(i) => setInputSearch(i.target.value)}
                    />
                </span>
                <ButtonContainer>
                    <Button text="Buscar" onClick={() => getDataListCategoryFilterDto({ nome: inputSearch }, { page: 0, size: queryPage.size })}></Button>
                </ButtonContainer>
                </InputWrapper>
            </SearchContainer>

            <SearchContainer>
                <InputWrapper style={{ justifyContent: 'start', padding: '0 0 12px' }}>
                <ButtonContainer style={{ width: 'inherit' }}>
                    <Button sizeText={13} text="Listar categoria" style={{ margin: '10px' }} onClick={() => { getDataListCategoryFilterDto(filterField, { page: 0, size: 6 }), setInputSearch("") } }></Button>
                    <Button sizeText={13} text="Nova categoria" style={{ margin: '10px' }} onClick={() => setVisible(true) }></Button>
                    <CategoryModal visible={visible} onHide={() => { setVisible(!visible) }} />
                </ButtonContainer>
                </InputWrapper>
            </SearchContainer>

            {/* @ts-ignore */}
            <DataTableCustom values={dataContent.content} columns={columnsDataTable} />
            <ReactPaginate
                forcePage={queryPage.page - 1}
                data-content="My Content"
                containerClassName={"Pagination"}
                marginPagesDisplayed={0}
                pageRangeDisplayed={4}
                pageCount={dataContent.totalPages ? dataContent.totalPages : 0}            
                previousLabel={"<"}
                nextLabel={">"}
                onPageChange={(page) => getDataListCategoryFilterDto({ nome: inputSearch }, { page: page.selected, size: queryPage.size })}
            />
        </Wrapper>
    );
}

export const getServerSideProps: GetServerSideProps = async (ctx) => {
    const { resolvedUrl, query } = ctx;
    const api = getApi(ctx);
    let filtro = { nome: query.search || "" }

    
    let language = parseCookies(ctx).language || "pt";
    if (ctx.locale != language && language) {
        return Util.redirectToSelectedLanguage(ctx, resolvedUrl);
    }
    
    const categories = await api.post(`WS/Categoria/Filtrar/Pageable`, filtro, {
        params: {
            page: 0,
            size: 6,
            sortDirection: 'ASC',
            sortField: 'nome'
        }
    })

    return {
        props: {
            ...(await serverSideTranslations(ctx.locale as string, ["common"])),
            listCategories: categories.data
        },
    };
};