import Head from "next/head";
import { useEffect, useRef, useState } from "react";
import HomeSearchBar from "../../app/components/HomeSearchBar";
import Loading from "../../app/components/Loading";
import { AdventureType } from "../../core/models/enums/AdventureType";
import { AdventureListDto } from "../../core/models/interfaces/Adventure/AdventureListDto";
import { Util } from "../../core/utils/util";
import { GetServerSideProps } from "next";
import useMap from "../../core/hooks/useMap";
import ReactDOMServer from "react-dom/server";
import MapCard, { Translations } from "../../app/components/MapCard";
import MarkerClusterer from "@googlemaps/markerclustererplus";
import { useGeolocationContext } from "../../core/contexts/GeolocationContext";
import api from "../../core/services/api";
import { custlerStyles } from "../../_constants";
import styled from "styled-components";
import { transparentize } from "polished";
import { parseCookies } from "nookies";
import { serverSideTranslations } from "next-i18next/serverSideTranslations";
import { useTranslation } from "next-i18next";
import nextI18NextConfig from "../../../next-i18next.config.js";
import { AdventureOrder } from "../../core/models/enums/AdventureOrder";
import useMasks from "../../core/hooks/useMasks";

interface MapProps {}

interface MapSearchParameters {
  search?: string | string[] | null;
  activeElement?: AdventureType | null;
  price?: AdventureType | string;
  order?: AdventureType | string;
  distance?: number;
  oldDistance?: number;
  location?: string;
  latitude?: number;
  longitude?: number;
}

function Map() {
  const [parameters, setParameters] = useState<MapSearchParameters>({
    search: "",
    activeElement: null,
    price: "",
    order: "",
    distance: 60,
    location: "",
    latitude: undefined,
    longitude: undefined,
    oldDistance: 60,
  });
  const { lastLocation } = useGeolocationContext();
  const { t } = useTranslation("common");
  const [ready, setReady] = useState(false);
  const { map, mapRef, circleRef, loadMap, createCircle, setMapCenter, getPlaceDetail } = useMap();
  const [markerCluster, setMarkerCluster] = useState<MarkerClusterer>();
  const markers = useRef<google.maps.Marker[]>([]);

  const { unMaskedMoney } = useMasks();

  // ----Map
  const createMarkers = (map: google.maps.Map, adventures: AdventureListDto[]) => {
    adventures?.forEach((adventure) => {
      const marker = new window.google.maps.Marker({
        position: { lat: adventure.latitude, lng: adventure.longitude },
        title: adventure.titulo,
        icon: Util.getPinImageByAdventureType(adventure.tipo),
        map: map,
      });

      addMarkerListener(marker, adventure);
      markers.current.push(marker);
    });

    const cluster = new MarkerClusterer(map, markers?.current, {
      styles: custlerStyles,
    });

    setMarkerCluster(cluster);
  };

  const removerMarkers = () => {
    if (markerCluster) {
      markerCluster.clearMarkers();
    }

    markers.current?.forEach((marker) => {
      marker.setMap(null);
    });

    markers.current = [];
  };

  const addMarkerListener = (marker: google.maps.Marker, adventure: AdventureListDto) => {
    var infoWindow: google.maps.InfoWindow = new google.maps.InfoWindow();

    marker.addListener("click", () => {
      map().panTo(marker.getPosition() as google.maps.LatLng);

      setTimeout(() => {
        infoWindow.setPosition(marker.getPosition());
        infoWindow.setOptions({ minWidth: 250, maxWidth: 300 });
        infoWindow.setContent(
          ReactDOMServer.renderToString(<MapCard adventure={adventure} t={getAdventuresTranslation(adventure)} />)
        );
        infoWindow.open(map(), marker);

        addInfoWindowListener(infoWindow, adventure, marker);
      }, 700);
    });
  };

  const addInfoWindowListener = (infoWindow: google.maps.InfoWindow, adventure: AdventureListDto, marker: any) => {
    google.maps.event.addListener(infoWindow, "domready", function () {
      let adventureCard = document.getElementById("adventure-" + adventure.id);

      if (adventureCard) {
        adventureCard.onclick = function () {
          window.open(`/adventure-details/${adventure.id}/${adventure?.slug}`, "_blank");
        };
      }
    });
  };

  const handleSearchAventure = (event?: React.KeyboardEvent<HTMLInputElement>) => {
    if (event) {
      if (event.key != "Enter") {
        return;
      }
    }
    removerMarkers();
    filterAdventures();
  };

  // -----Map

  // ----Adventures

  const getAdventuresTranslation = (adventure: AdventureListDto): Translations => {
    return {
      adventureType: t(Util.getNameByAdventureType(adventure.tipo)),
      perPerson: t("perPerson"),
      where: t("where"),
    };
  };

  const filterAdventures = async () => {
    const { search, activeElement, order, price, distance, latitude, longitude, oldDistance } = parameters;

    const body = {
      tipoAventura: activeElement,
      buscaLivre: search,
      filtrarSomenteDisponivel: true,
      preco: price && price != "" ? unMaskedMoney(price) : null,
      distancia: distance,
      latitude: latitude || lastLocation.lat,
      longitude: longitude || lastLocation.lng,
    };

    const adventures = await api.post(
      `/WS/Aventura/Todas/Lista?${Util.mountAdventureOrder(order as AdventureOrder)}`,
      body
    );

    createMarkers(map(), adventures.data);

    setMapCenter({ lat: body.latitude, lng: body.longitude });

    if (distance) {
      createCircle({ lat: body.latitude, lng: body.longitude }, distance);
    } else {
      circleRef.current?.setMap(null);
    }
    setTimeout(() => {
      map().panTo(new google.maps.LatLng(body.latitude, body.longitude));
      if (distance && oldDistance != distance) {
        if (distance <= 50) {
          map().setZoom(10);
        } else if (distance <= 100 && distance > 50) {
          map().setZoom(7);
        } else {
          map().setZoom(6);
        }
        parameters.oldDistance = distance;
      }
    }, 500);
  };

  const clearFilters = () => {
    parameters.order = "";
    parameters.activeElement = null;
    parameters.search = "";
    parameters.price = "";
    parameters.distance = 60;
    parameters.latitude = undefined;
    parameters.longitude = undefined;
    parameters.location = "";

    handleSearchAventure();
  };

  const setActiveElement = (newElement: AdventureType | null | undefined) => {
    setParameters({ ...parameters, activeElement: newElement });
    parameters.activeElement = newElement;
    handleSearchAventure();
  };

  const setSearch = (text?: string | string[] | null) => {
    setParameters({ ...parameters, search: text });
  };

  const setPrice = (text?: string) => {
    setParameters({ ...parameters, price: text });
  };

  const setDistance = (text?: string) => {
    setParameters({ ...parameters, distance: Number(text) });
  };

  const setLocation = (text?: string) => {
    setParameters({ ...parameters, location: text });
  };

  const filters = {
    price: {
      value: parameters.price as string,
      visible: true,
      setMethod: setPrice,
    },
    distance: {
      value: String(parameters.distance),
      visible: true,
      setMethod: setDistance,
    },
    location: {
      value: String(parameters.location),
      visible: true,
      setMethod: setLocation,
    },
  };

  const reposisitionMap = (placeId: string) => {
    if (placeId) {
      getPlaceDetail(placeId, (place) => {
        if (place) {
          const position = {
            lat: place.geometry?.location?.lat() as number,
            lng: place.geometry?.location?.lng() as number,
          };

          parameters.latitude = position.lat;
          parameters.longitude = position.lng;
          parameters.location = place.formatted_address;
          handleSearchAventure();
        }
      });
    }
  };

  // ----Adventures

  useEffect(() => {
    loadMap(lastLocation, () => {
      setReady(true);
      filterAdventures();
    });
  }, []);

  return (
    <>
      <Head>
        <title>Relty - {t("exploreInMap")}</title>
      </Head>

      {/* ------- Search */}
      <HomeSearchBar
        activeElement={parameters.activeElement}
        setActiveElement={setActiveElement}
        search={parameters.search || null}
        setSearch={setSearch}
        handleSearchAventure={handleSearchAventure}
        advancedFilters={filters}
        clearFilters={clearFilters}
        reposisitionMap={reposisitionMap}
      />

      {/* ------- Search */}

      {ready ? <MapContainer ref={mapRef}></MapContainer> : <Loading width={100} height={100} />}
    </>
  );
}

export default Map;

const MapContainer = styled.div`
  width: 100%;
  height: 600px;
  margin-top: 16px;
  border-radius: ${(p) => p.theme.borderRadius};
  box-shadow: 0 3px 10px ${(p) => transparentize(0.7, p.theme.pageForeground)};

  @media screen and (max-width: 767px) {
    height: 500px;
  }
`;

export const getServerSideProps: GetServerSideProps<MapProps> = async (ctx) => {
  const { resolvedUrl } = ctx;

  let language = parseCookies(ctx).language || "pt";
  if (ctx.locale != language && language) {
    return Util.redirectToSelectedLanguage(ctx, resolvedUrl);
  }

  return {
    props: {
      ...(await serverSideTranslations(ctx.locale as string, ["common"], nextI18NextConfig)),
    },
  };
};
