import { GetServerSideProps, GetStaticProps } from "next";
import Head from "next/head";
import CardAdventure from "../../app/components/CardAdventure";
import { AdventureListDto } from "../../core/models/interfaces/Adventure/AdventureListDto";
import { Util } from "../../core/utils/util";
import { CardContainer, CardsGrid, NoData, SearchContainer } from "./styles";
import ReactPaginate from "react-paginate";
import Router from "next/router";
import HomeSearchBar from "../../app/components/HomeSearchBar";
import { AdventureType } from "../../core/models/enums/AdventureType";
import { getApi } from "../../core/services/apiServerSide";
import { useState } from "react";
import { parseCookies } from "nookies";
import { serverSideTranslations } from "next-i18next/serverSideTranslations";
import { useTranslation } from "next-i18next";
import { AdventureOrder } from "../../core/models/enums/AdventureOrder";
import useMasks from "../../core/hooks/useMasks";
interface HomeProps {
  adventures?: AdventureListDto[];
  totalPages?: number;
  adventureType?: AdventureType | null;
  search?: string | string[] | null;
  page: number;
  price?: string | string[] | null;
  order?: string | null;
}

interface HomeSearchParameters {
  search?: string | string[] | null;
  activeElement?: AdventureType | null;
  page: number;
  price?: string | string[] | null;
  order?: string | null;
}

const Home = (props: HomeProps) => {
  const { adventures, totalPages } = props;
  const { unMaskedMoney, money } = useMasks();
  const { t } = useTranslation("common");
  const [parameters, setParameters] = useState<HomeSearchParameters>({
    page: props.page,
    search: props.search,
    activeElement: props.adventureType,
    price: props.price ? money(parseFloat(String(props.price)).toFixed(2)).maskedValue : props.price,
    order: props.order,
  });

  const handleSearchAventure = (event?: React.KeyboardEvent<HTMLInputElement>, page?: number) => {
    if (event) {
      if (event.key != "Enter") {
        return;
      } else {
        setParameters({ ...parameters, page: 1 });
      }
    }

    const { search, activeElement, price, order } = parameters;

    let finalUrl = Util.mountAdventureFilterParams(
      search as string,
      activeElement,
      String(unMaskedMoney(price as string)),
      order as AdventureOrder
    );
    let url = `/home?page=${page || 1}` + finalUrl;

    Router.replace(url);
  };

  const hrefBuilder = (page?: number): string => {
    const { search, activeElement, price, order } = parameters;
    return (
      `/home?page=${page || 1}` +
      Util.mountAdventureFilterParams(
        search as string,
        activeElement,
        String(unMaskedMoney(price as string)),
        order as AdventureOrder
      )
    );
  };

  const clearFilters = () => {
    parameters.order = "";
    parameters.activeElement = null;
    parameters.search = "";
    parameters.price = null;
    parameters.page = 1;

    handleSearchAventure();
  };

  const setActiveElement = (newElement: AdventureType | null | undefined) => {
    setParameters({ ...parameters, activeElement: newElement });
    parameters.activeElement = newElement;
    handleSearchAventure();
  };

  const setSearch = (text?: string | string[] | null) => {
    setParameters({ ...parameters, search: text });
  };

  const setPrice = (text?: string) => {
    setParameters({ ...parameters, price: text });
  };

  const setOrder = (text?: string) => {
    setParameters({ ...parameters, order: text });
    parameters.order = text;
    handleSearchAventure(undefined, 1);
  };

  const filters = {
    price: {
      value: parameters.price as string,
      visible: true,
      setMethod: setPrice,
    },
    order: {
      value: parameters.order as AdventureOrder,
      visible: true,
      setMethod: setOrder,
    },
  };

  return (
    <div>
      <Head>
        <title>Relty - Home</title>
        <meta name="description" content="Relty ativades físicas e aventuras" />
        <link rel="icon" href="/assets/icon.png" />
      </Head>

      <SearchContainer>
        {/* ------- Search */}
        <HomeSearchBar
          activeElement={parameters.activeElement}
          setActiveElement={setActiveElement}
          search={parameters.search || null}
          setSearch={setSearch}
          handleSearchAventure={handleSearchAventure}
          advancedFilters={filters}
          clearFilters={clearFilters}
        />
        {/* ------- Search */}
      </SearchContainer>

      {adventures && adventures.length > 0 ? (
        <>
          <CardsGrid>
            {adventures?.map((item) => (
              <CardContainer key={item.id}>
                <CardAdventure adventure={item} />
              </CardContainer>
            ))}
          </CardsGrid>

          <ReactPaginate
            forcePage={parameters.page - 1}
            data-content="My Content"
            containerClassName={"Pagination"}
            pageCount={totalPages ? totalPages : 0}
            marginPagesDisplayed={0}
            pageRangeDisplayed={3}
            previousLabel={"<"}
            nextLabel={">"}
            hrefBuilder={(page) => hrefBuilder(page)}
            onPageChange={(page) => handleSearchAventure(undefined, page.selected + 1)}
          />
        </>
      ) : (
        <NoData>
          <h2>{t("noResults")}</h2>
        </NoData>
      )}
    </div>
  );
};

export default Home;

export const getServerSideProps: GetServerSideProps<HomeProps> = async (ctx) => {
  const { query, res, resolvedUrl } = ctx;

  const { page: _page, adventureType, search, price: _price, order } = query;
  const page = Number(_page);
  const price = Number(_price);

  let language = parseCookies(ctx).language || "pt";

  if (ctx.locale != language && language) {
    return Util.redirectToSelectedLanguage(ctx, resolvedUrl);
  }

  const api = getApi(ctx);

  if (adventureType && Util.getValidAdventureType(adventureType) === null) {
    return Util.sendToHomePage(res, page);
  }

  if (price && (isNaN(price) || price < 0)) {
    return Util.sendToHomePage(res, page);
  }

  if (isNaN(page) || page < 1) {
    return Util.sendToHomePage(res);
  }

  let body = {
    tipoAventura: Util.getValidAdventureType(adventureType),
    buscaLivre: search ? search : null,
    filtrarSomenteDisponivel: true,
    preco: price ? price?.toString() : null,
  };

  const adventures = await api.post(
    `/WS/Aventura/Filtrar?page=${page - 1}&size=12 ${Util.mountAdventureOrder(String(order) as AdventureOrder)}`,
    body
  );

  return {
    props: {
      ...(await serverSideTranslations(ctx.locale as string, ["common"])),
      adventures: adventures.data.content,
      totalPages: adventures.data.totalPages,
      adventureType: Util.getValidAdventureType(adventureType),
      search: search || null,
      page: page,
      order: (order as string) || null,
      price: price ? price?.toString() : null,
    },
  };
};
