import { GetServerSideProps } from "next";
import Head from "next/head";
import CardAdventure from "../../app/components/CardAdventure";
import { AdventureListDto } from "../../core/models/interfaces/Adventure/AdventureListDto";
import { Util } from "../../core/utils/util";
import { CardContainer, CardsGrid, NoData, SearchContainer } from "./styles";
import ReactPaginate from "react-paginate";
import Router from "next/router";
import HomeSearchBar from "../../app/components/HomeSearchBar";
import { AdventureType } from "../../core/models/enums/AdventureType";
import { getApi } from "../../core/services/apiServerSide";
import { useCallback, useEffect, useState } from "react";
import i18next from "i18next";
import HeaderProfile from "../../app/components/HeaderProfile";
import { User } from "../../core/models/interfaces/User/User";
import { PublicProfileDto } from "../../core/models/interfaces/Client/PublicProfileDto";
import { useUserContext } from "../../core/contexts/UserContext";
import { parseCookies } from "nookies";
import { serverSideTranslations } from "next-i18next/serverSideTranslations";
import { useTranslation } from "next-i18next";

interface FavoritosProps {
  adventures?: AdventureListDto[];
  totalPages?: number;
  adventureType?: AdventureType | null;
  search?: string | string[] | null;
  page: number;
  publicProfile?: PublicProfileDto;
}

interface FavoritosSearchParameters {
  search?: string | string[] | null;
  activeElement?: AdventureType | null;
  page: number;
}

const Favoritos = (props: FavoritosProps) => {
  const { adventures, totalPages, publicProfile } = props;
  const { t } = useTranslation("common");
  const { updateUser, user } = useUserContext();

  const checkUpdateUser = useCallback(async () => {
    if (!user.clienteId) {
      updateUser;
    }
  }, [updateUser, user.clienteId]);

  useEffect(() => {
    checkUpdateUser();
  }, [checkUpdateUser]);

  const [parameters, setParameters] = useState<FavoritosSearchParameters>({
    page: props.page,
    search: props.search,
    activeElement: props.adventureType,
  });

  const handleSearchAventure = (event?: React.KeyboardEvent<HTMLInputElement>, page?: number) => {
    if (event) {
      if (event.key != "Enter") {
        return;
      } else {
        setParameters({ ...parameters, page: 1 });
      }
    }

    const { search, activeElement } = parameters;

    let url = `/favorites?page=${page || 1}` + Util.mountAdventureFilterParams(search as string, activeElement);

    Router.replace(url);
  };

  const hrefBuilder = (page?: number): string => {
    const { search, activeElement } = parameters;
    return `/favorites?page=${page || 1}` + Util.mountAdventureFilterParams(search as string, activeElement);
  };

  const setActiveElement = (newElement: AdventureType | null | undefined) => {
    setParameters({ ...parameters, activeElement: newElement });
    parameters.activeElement = newElement;
    handleSearchAventure();
  };

  const setSearch = (text?: string | string[] | null) => {
    setParameters({ ...parameters, search: text });
  };

  return (
    <div>
      <Head>
        <title>Relty - {t("favoriteAdventures")}</title>
      </Head>

      <SearchContainer>
        <HeaderProfile publicProfile={publicProfile} />

        {/* ------- Search */}
        <HomeSearchBar
          activeElement={parameters.activeElement}
          setActiveElement={setActiveElement}
          search={parameters.search || null}
          setSearch={setSearch}
          handleSearchAventure={handleSearchAventure}
        />
        {/* ------- Search */}
      </SearchContainer>

      {adventures && adventures.length > 0 ? (
        <>
          <CardsGrid>
            {adventures?.map((item) => (
              <CardContainer key={item.id}>
                <CardAdventure adventure={item} />
              </CardContainer>
            ))}
          </CardsGrid>

          <ReactPaginate
            forcePage={parameters.page - 1}
            containerClassName={"Pagination"}
            pageCount={totalPages ? totalPages : 0}
            marginPagesDisplayed={0}
            pageRangeDisplayed={3}
            previousLabel={"<"}
            nextLabel={">"}
            hrefBuilder={(page) => hrefBuilder(page)}
            onPageChange={(page) => handleSearchAventure(undefined, page.selected + 1)}
          />
        </>
      ) : (
        <NoData>
          <h2>{t("noResults")}</h2>
        </NoData>
      )}
    </div>
  );
};

export default Favoritos;

export const getServerSideProps: GetServerSideProps<FavoritosProps> = async (ctx) => {
  const { query, res, resolvedUrl } = ctx;
  const { page: _page, adventureType, search } = query;
  const page = Number(_page);

  const api = getApi(ctx);

  //redirect
  if (!parseCookies(ctx).access_token) {
    res.statusCode = 302;
    return {
      redirect: {
        destination: "/home?page=1",
        permanent: false,
      },
      props: {},
    };
  }

  let language = parseCookies(ctx).language || "pt";
  if (ctx.locale != language && language) {
    return Util.redirectToSelectedLanguage(ctx, resolvedUrl);
  }

  if (adventureType && Util.getValidAdventureType(adventureType) === null) {
    return Util.sendToFavoritesPage(res, page);
  }

  if (isNaN(page) || page < 1) {
    return Util.sendToFavoritesPage(res);
  }

  let savedUser: User = JSON.parse(parseCookies(ctx).user);

  let body = {
    aventura: {
      tipoAventura: Util.getValidAdventureType(adventureType),
      buscaLivre: search ? search : null,
    },
    clienteId: savedUser.clienteId,
  };

  const publicProfile = await api.get(`/WS/Cliente/Perfil`);

  const adventures = await api.post(`/WS/ClienteFavorito/Aventura/Filtrar?page=${page - 1}&size=12`, body);

  return {
    props: {
      ...(await serverSideTranslations(ctx.locale as string, ["common"])),
      adventures: adventures.data.content,
      totalPages: adventures.data.totalPages,
      adventureType: Util.getValidAdventureType(adventureType),
      search: search || null,
      page: page,
      publicProfile: publicProfile.data,
    },
  };
};
