import { GetServerSideProps } from "next";
import Head from "next/head";
import CardAdventureUser from "../../app/components/CardAdventureUser";
import { AdventureListDto } from "../../core/models/interfaces/Adventure/AdventureListDto";
import { Util } from "../../core/utils/util";
import { CardContainer, CardsGrid, NoData, SearchContainer, TextFont, ContainerText } from "./styles";
import ReactPaginate from "react-paginate";
import Router from "next/router";
import HomeSearchBar from "../../app/components/HomeSearchBar";
import { AdventureType } from "../../core/models/enums/AdventureType";
import { getApi } from "../../core/services/apiServerSide";
import { useState } from "react";
import { parseCookies } from "nookies";
import { serverSideTranslations } from "next-i18next/serverSideTranslations";
import { useTranslation } from "next-i18next";
import { AdventureOrder } from "../../core/models/enums/AdventureOrder";
import { User } from "../../core/models/interfaces/User/User";

interface AdventuresProps {
    adventures?: AdventureListDto[];
    totalPages?: number;
    adventureType?: AdventureType | null;
    search?: string | string[] | null;
    page: number;
}

interface AdventuresSearchParameters {
    search?: string | string[] | null;
    activeElement?: AdventureType | null;
    page: number;
}

export default function Adventures(props: AdventuresProps){
    const { adventures, totalPages } = props;
    const { t } = useTranslation("common");
    const [parameters, setParameters] = useState<AdventuresSearchParameters>({
        page: props.page,
        search: props.search,
        activeElement: props.adventureType,
    });

    const handleSearchAventure = (event?: React.KeyboardEvent<HTMLInputElement>, page?: number) => {
        const { search, activeElement } = parameters;

        if (event) {
            if (event.key != "Enter") {
                return;
            } else {
                setParameters({ ...parameters, page: 1 });
            }
        }

        let finalUrl = Util.mountAdventureFilterParams(search as string, activeElement);
        let url = `/adventures?page=${page || 1}` + finalUrl;
        Router.replace(url);
    };

    const hrefBuilder = (page?: number): string => {
        const { search, activeElement } = parameters;
        return (
            `/adventures?page=${page || 1}` + Util.mountAdventureFilterParams(search as string, activeElement)
        );
    };

    const setActiveElement = (newElement: AdventureType | null | undefined) => {
        setParameters({ ...parameters, activeElement: newElement });
        parameters.activeElement = newElement;
        handleSearchAventure();
    };

    const setSearch = (text?: string | string[] | null) => {
        setParameters({ ...parameters, search: text });
    };

return (
    <div>
        <Head>
            <title>Relty - Minhas Aventuras</title>
            <meta name="description" content="Relty ativades físicas e aventuras" />
            <link rel="icon" href="/assets/icon.png" />
        </Head>

        <ContainerText>
            <TextFont fontSize={24} fontWeight={600}> Minhas aventuras </TextFont>
            <TextFont fontSize={24} > Aqui estão todas as suas aventuras reservadas ou já realizadas. </TextFont>
        </ContainerText>

        <SearchContainer>
            {/* ------- Search */}
            <HomeSearchBar
                activeElement={parameters.activeElement}
                setActiveElement={setActiveElement}
                search={parameters.search || null}
                setSearch={setSearch}
                handleSearchAventure={handleSearchAventure}
            />
            {/* ------- Search */}
        </SearchContainer>

        {adventures && adventures.length > 0 ? (
            <>
            <CardsGrid>
                {adventures?.map((item, index) => (
                <CardContainer key={index}>
                    <CardAdventureUser adventure={item} />
                </CardContainer>
                ))}
            </CardsGrid>
            <ReactPaginate
                forcePage={parameters.page - 1}
                data-content="My Content"
                containerClassName={"Pagination"}
                pageCount={totalPages ? totalPages : 0}
                marginPagesDisplayed={0}
                pageRangeDisplayed={3}
                previousLabel={"<"}
                nextLabel={">"}
                hrefBuilder={(page) => hrefBuilder(page)}
                onPageChange={(page) => handleSearchAventure(undefined, page.selected + 1)}
            />
            </>
        ) : (
            <NoData>
                <h2>{t("noResults")}</h2>
            </NoData>
        )}
    </div>
    );
};


export const getServerSideProps: GetServerSideProps<AdventuresProps> = async (ctx) => {
    const { query, res, resolvedUrl } = ctx;
    const { page: _page, adventureType, search, order } = query;
    const page = Number(_page);
    const api = getApi(ctx);

    let language = parseCookies(ctx).language || "pt";
    let savedUser: User = JSON.parse(parseCookies(ctx).user);

    if (ctx.locale != language && language) {
        return Util.redirectToSelectedLanguage(ctx, resolvedUrl);
    }

    if (adventureType && Util.getValidAdventureType(adventureType) === null) {
        return Util.sendToUserAdventures(res, page);
    }

    if (isNaN(page) || page < 1) {
        return Util.sendToUserAdventures(res);
    }

    let body = {
        aventura: {
            tipoAventura: Util.getValidAdventureType(adventureType),
            buscaLivre: search ? search : null,
        },
        cliente: {
            id: savedUser.clienteId
        }
    };

    const adventures = await api.post(
        `/WS/ClienteAventura/Aventura/Filtrar/?page=${page - 1}&size=12 ${Util.mountAdventureOrder(String(order) as AdventureOrder)}`,
        body
    );

return {
    props: {
            ...(await serverSideTranslations(ctx.locale as string, ["common"])),
            adventures: adventures.data.content,
            totalPages: adventures.data.totalPages,
            adventureType: Util.getValidAdventureType(adventureType),
            search: search || null,
            page: page,
            order: (order as string) || null
        },
    };
};
