import { ServerResponse } from "http";
import { GetServerSidePropsContext } from "next";
import { parseCookies } from "nookies";
import { ParsedUrlQuery } from "querystring";
import { AdventureOrder } from "../models/enums/AdventureOrder";
import { AdventureType } from "../models/enums/AdventureType";

function showSucessToast(ref: any, message: string) {
  if (ref.current && message) {
    ref.current?.show({
      severity: "success",
      summary: message,
      detail: "",
      contentClassName: "",
    });
  }
}

function showWarnToast(ref: any, message: string) {
  if (ref.current && message) {
    ref.current?.show({
      severity: "warn",
      summary: message,
      detail: "",
      contentClassName: "",
    });
  }
}

function showErrorToast(ref: any, message: string) {
  if (ref.current && message) {
    ref.current?.show({
      severity: "error",
      summary: message,
      detail: "",
      contentClassName: "",
    });
  }
}

function sendToAnyPage(res: ServerResponse, path: string) {
  res.statusCode = 302;
  res.setHeader("Location", path);
  return { props: {} };
}

function redirectToSelectedLanguage(ctx: GetServerSidePropsContext<ParsedUrlQuery>, baseUrl: string) {
  const { res } = ctx;

  res.statusCode = 302;
  const language = `/${parseCookies(ctx).language || "pt"}`;

  let url = `${!language || language == "/pt" || language.toLowerCase() == "/pt-br" ? "" : language}${baseUrl}`;

  return {
    redirect: {
      destination: url,
      permanent: false,
    },
    props: {},
  };
}

function sendToHomePage(res: ServerResponse, page = 1) {
  res.statusCode = 302;
  res.setHeader("Location", `/home?page=${page}`);
  return { props: { page } };
}

function sendToSupplierAdventures(res: ServerResponse, page = 1) {
  res.statusCode = 302;
  res.setHeader("Location", `/profiles/supplier/adventures?page=${page}`);
  return { props: { page } };
}

function sendToUserAdventures(res: ServerResponse, page = 1) {
  res.statusCode = 302;
  res.setHeader("Location", `/adventures?page=${page}`);
  return { props: { page } };
}

function sendToFavoritesPage(res: ServerResponse, page = 1) {
  res.statusCode = 302;
  res.setHeader("Location", `/favorites?page=${page}`);
  return { props: { page } };
}

function getColorByAdventureType(type: string): string {
  switch (type) {
    case AdventureType.A:
      return "#58BCFF";

    case AdventureType.G:
      return "#00CBC5";

    case AdventureType.T:
      return "#D15E34";

    default:
      return "#006993";
  }
}

function getColorByDifficulty(type: string): string {
  switch (type) {
    case "F":
      return "#4CD964";

    case "M":
      return "#FBCB20";

    case "D":
      return "#D5403B";

    default:
      return "#FFFFFF";
  }
}

function getNameByAdventureType(type: string): string {
  switch (type) {
    case AdventureType.A:
      return "air";

    case AdventureType.G:
      return "water";

    case AdventureType.T:
      return "earth";

    default:
      return "";
  }
}

function getNameByAdventureDifficulty(difficulty: string) {
  switch (difficulty) {
    case "F":
      return "easy";

    case "M":
      return "medium";

    case "D":
      return "hard";

    default:
      return "";
  }
}

function getValidAdventureType(type?: any): AdventureType | null {
  switch (type) {
    case "A":
      return AdventureType.A;

    case "G":
      return AdventureType.G;

    case "T":
      return AdventureType.T;

    default:
      return null;
  }
}

function mountAdventureFilterParams(
  search?: string,
  adventureType?: AdventureType | null,
  price?: string,
  mostPracticed?: AdventureOrder
): string {
  let newUrl = "";

  if (adventureType) {
    newUrl = newUrl + "&adventureType=" + adventureType;
  }

  if (search && search != "") {
    newUrl = newUrl + "&search=" + search;
  }

  if (price && price != "") {
    newUrl = newUrl + "&price=" + price;
  }

  if (mostPracticed && mostPracticed == AdventureOrder.M) {
    newUrl = newUrl + "&order=" + mostPracticed;
  }
  return newUrl;
}

function mountAdventureOrder(order?: AdventureOrder): string {
  let newUrl = "";

  if (order && order == AdventureOrder.M) {
    newUrl = newUrl + "&sortField=analise.quantidadeVezesPraticada&sortDirection=DESC";
  }
  return newUrl;
}

/**
 *
 * A função deve ser chamada sempre dentro de um useEffect ao montar o componente por não funcionar do lado do servidor
 */

function isMobile() {
  try {
    if (
      navigator.userAgent.match(/Android/i) ||
      navigator.userAgent.match(/webOS/i) ||
      navigator.userAgent.match(/iPhone/i) ||
      navigator.userAgent.match(/iPad/i) ||
      navigator.userAgent.match(/iPod/i) ||
      navigator.userAgent.match(/BlackBerry/i) ||
      navigator.userAgent.match(/Windows Phone/i)
    ) {
      return true; // está utilizando celular
    } else {
      return false; // não é celular
    }
  } catch (error) {
    return false;
  }
}

const isAndroid = () => {
  try {
    if (navigator.userAgent.match(/Android/i)) {
      return true; // está utilizando android
    } else {
      return false; // não é android
    }
  } catch (error) {}
};

const getPinImageByAdventureType = (type?: AdventureType) => {
  switch (type) {
    case AdventureType.A:
      return "/assets/airpin.svg";

    case AdventureType.G:
      return "/assets/waterpin.svg";

    case AdventureType.T:
      return "/assets/earthpin.svg";

    default:
      return "/assets/normalpin.svg";
  }
};

export const Util = {
  //toasts
  showErrorToast,
  showWarnToast,
  showSucessToast,
  //adventures
  getColorByAdventureType,
  getColorByDifficulty,
  getNameByAdventureType,
  getNameByAdventureDifficulty,
  getValidAdventureType,
  mountAdventureFilterParams,
  mountAdventureOrder,
  getPinImageByAdventureType,
  //adventures

  isMobile,
  isAndroid,

  //redirects
  sendToHomePage,
  sendToSupplierAdventures,
  sendToFavoritesPage,
  sendToUserAdventures,
  sendToAnyPage,
  redirectToSelectedLanguage,
};
