import { AxiosResponse } from "axios";
import router from "next/router";
import { parseCookies, setCookie } from "nookies";
import { languages } from "../contexts/PreferencesContext";
import { useUserContext } from "../contexts/UserContext";
import { User } from "../models/interfaces/User/User";
import { UserEditDto } from "../models/interfaces/User/UserEditDto";
import { UserRegisterDtoAdmin } from "../models/interfaces/User/UserRegisterDtoAdmin";
import api from "../services/api";

export default function useUser() {
  const { user, isAuthenticated, setUser } = useUserContext();

  const getDataEditing = async (): Promise<AxiosResponse<UserEditDto>> => {
    return await api.get(`/WS/UsuarioRelty/${user.id}/Dto/Cliente`);
  };

  const saveDataEditing = async (userEditDto: UserEditDto): Promise<AxiosResponse<User>> => {
    return await api.post(`/WS/UsuarioRelty/Editar`, userEditDto);
  };

  const alterProfile = async (profile: string) => {
    await api.post(`WS/UsuarioRelty/AlterarPerfil/${profile}`).then((result) => {
      let updatedUser = { ...user };
      updatedUser.perfilAtual = profile;
      updatedUser.clienteId = result.data.clienteId;
      updatedUser.estabelecimentoId = result.data.estabelecimentoId;
      updateUserInCookie(updatedUser);
    });
  };

  const changeUserPassword = async (id: number, password: string): Promise<AxiosResponse<any>> => {
    return await api.post(`/WS/UsuarioRelty/AlterarSenha`, { id: id, senha: password });
  }

  const turnUserIntoAdmin = async (id: number): Promise<AxiosResponse<any>> => {
    return await api.post(`/WS/UsuarioRelty/AlternarAdm/${id}`, {});
  }

  const createAdminUser = async (body: UserRegisterDtoAdmin): Promise<AxiosResponse<any>> => {
    return await api.post("/WS/UsuarioRelty/Criar/Admin", body);
  }

  const updateUserInCookie = (user: User) => {
    setUser(user);
    setCookie(undefined, "user", JSON.stringify(user), { path: "/", maxAge: 30 * 24 * 60 * 60 * 2 });

    redirectByUserProfile(user);
  };

  const getUpdatedProfile = async (id: number): Promise<AxiosResponse<User>> => {
    return await api.get(`WS/UsuarioRelty/Dados/${id}`);
  };

  ///Esses metodos usam o cookie por questão de serem chamados no contexto
  // de Preferencias e não ter acesso ao userContext
  const updateUserLanguage = async (language: languages) => {
    if (parseCookies(undefined, { path: "/" }).access_token) {
      await api.post("/WS/UsuarioRelty/Linguagem", { linguagem: language });
    }
  };

  const updateOneSignalToken = async (token: string) => {
    if (parseCookies(undefined, { path: "/" }).access_token) {
      await api.post("/WS/UsuarioRelty/OneSignal", { onesignalToken: token });
    }
  };

  async function makeUserAdm(id: number): Promise<AxiosResponse<any>> {
    return await api.post(`/WS/UsuarioRelty/AlternarAdm/${id}`, null)
  }

  const redirectByUserProfile = (user: User) => {
    let pathname = window.location.pathname;

    if (pathname.includes("/profiles/supplier") && user.perfilAtual == "C") {
      router.replace("/profiles/user");
      return;
    }

    if (pathname.includes("/profiles/user") && user.perfilAtual == "E") {
      router.replace("/profiles/supplier");
      return;
    }
  };

  return {
    getDataEditing,
    saveDataEditing,
    alterProfile,
    changeUserPassword,
    turnUserIntoAdmin,
    createAdminUser,
    getUpdatedProfile,
    updateUserInCookie,
    updateUserLanguage,
    updateOneSignalToken,
    makeUserAdm
  };
}
