import i18next from "i18next";
import { useEffect, useRef } from "react";
import { useTheme } from "styled-components";
import { loadGoogleMaps, removeGoogleMaps } from "../utils/googleMaps";
import { useGeolocationContext } from "../contexts/GeolocationContext";

export default function useMap() {
  const mapRef = useRef<any>();
  const userMarker = useRef<google.maps.Marker>();
  const circleRef = useRef<google.maps.Circle>();
  const theme = useTheme();
  const { setLocalizationMaps } = useGeolocationContext()

  const loadMap = (position: { lat: number; lng: number }, callback: () => void) => {
    loadGoogleMaps(() => {
      callback();
      mapRef.current = new window.google.maps.Map(mapRef.current, {
        center: position,
        zoom: 7,
      });
      setMapCenter(position);
      map().setCenter(position);

      map().addListener('click', (event: any) => {
        setLocalizationMaps({ lat: event.latLng.lat(), lng: event.latLng.lng() })
        setMapCenter({ lat: event.latLng.lat(), lng: event.latLng.lng() })
      })

    });
  };

  const setMapCenter = (userPosition: { lat: number; lng: number }, legend = false, icon = "/assets/normalpin.svg") => {
    var infoWindow: google.maps.InfoWindow = new google.maps.InfoWindow();
    const marker = new window.google.maps.Marker({
      position: userPosition,
      title: i18next.t("yourLocation"),
      map: map(),
      icon: icon,
    });

    if (userMarker.current) {
      userMarker.current.setMap(null);
    }

    userMarker.current = marker;

    if (legend) {
      infoWindow.setPosition(marker.getPosition());
      infoWindow.setContent(
        `<div style='width:100%; text-align: center; font-family: Poppins'>${i18next.t("yourLocation")}</div>`
      );
      infoWindow.setOptions({});
      infoWindow.setOptions({ minWidth: 65 });
      infoWindow.open(map(), marker);

      setTimeout(() => {
        infoWindow.close();
      }, 5000);
    }
  };

  const createCircle = (position: { lat: number; lng: number }, radius: number) => {
    if (circleRef.current) {
      circleRef.current.setMap(null);
    }

    circleRef.current = new google.maps.Circle({
      strokeColor: theme.primaryBackground,
      strokeOpacity: 0.8,
      strokeWeight: 2,
      fillColor: theme.primaryBackground,
      fillOpacity: 0.35,
      map: map(),
      center: position,
      radius: radius * 1000,
    });
  };

  const searchPlaces = async (text: string): Promise<google.maps.places.AutocompletePrediction[]> => {
    console.log(text)
    let results = await new window.google.maps.places.AutocompleteService().getPlacePredictions({
      input: text,
      types: ["(cities)"],
    });
    return results.predictions;
  };

  const getPlaceDetail = async (
    placeId: string,
    callBack: (place: google.maps.places.PlaceResult | null) => void
  ): Promise<void> => {
    const placeService = new window.google.maps.places.PlacesService(map());

    let response = null;
    await placeService.getDetails(
      {
        placeId: placeId,
        fields: ["geometry", "formatted_address"],
      },
      (result) => {
        if (callBack) callBack(result);
      }
    );
  };

  const map = (): google.maps.Map => {
    return mapRef.current;
  };

  useEffect(() => {
    return () => {
      removeGoogleMaps();
    };
  }, []);

  return {
    mapRef,
    circleRef,
    map,
    setMapCenter,
    loadMap,
    createCircle,
    searchPlaces,
    getPlaceDetail
  };
}
