import React, { ElementType, useRef, useState } from "react";
import { Dialog } from "primereact/dialog";

import Button from "../Button";
import Input from "../CustomInput";
import i18next from "i18next";
import { Toast } from "primereact/toast";

import { Formik, Form } from "formik";
import * as Yup from "yup";
import useAuth from "../../../core/hooks/useAuth";
import { Util } from "../../../core/utils/util";
import { BadCredentialsException } from "../../../core/utils/exceptions";
import router from "next/router";
import Mail from "../SvgIcons/mail";
import Password from "../SvgIcons/password";
import { ContentWrapper, ButtonContainer, Text, ForgotPassword } from "./styles";
import axios from "axios";
import { baseURL } from "../../../core/services/apiServerSide";
import { useTheme } from "styled-components";
import { useTranslation } from "next-i18next";

export default function WithAuthModal(Component: ElementType) {
  const Wrapper = (props: any) => {
    const { t } = useTranslation("common");
    const [authModal, setAuthModal] = useState(false);
    const [loginActive, setLoginActive] = useState(false);
    const [forgotPassword, setForgotPassword] = useState(false);
    const { login } = useAuth();
    const theme = useTheme();

    const toast = useRef(null);

    const onHide = () => {
      setAuthModal(false);
      setLoginActive(false);
      setForgotPassword(false);
    };

    const setAuthModalOnLogin = () => {
      setAuthModal(true);
      setLoginActive(true);
    };

    const createAccount = () => {
      router.push("/register");
      onHide();
    };

    async function recoverPassword(email: string) {
      await axios
        .post(baseURL + "/WS/UsuarioRelty/RecuperarSenha", { email: email })
        .then((result) => {
          Util.showSucessToast(toast, t("passwordToEmail"));
          setForgotPassword(false);
        })
        .catch((error) => {
          if (error.response.status == 412) {
            Util.showErrorToast(toast, error.response.data?.descricao);
          } else {
            Util.showErrorToast(toast, t("onRecoverPasswordError"));
          }
        });
    }

    const initialValues = {
      email: "",
      password: "",
    };

    const onSubmit = async (values: any) => {
      try {
        await login(values.email, values.password);
        Util.showSucessToast(toast, t("successLogin"));
        onHide();

        setTimeout(() => {
          router.reload();
        }, 800);
      } catch (error) {
        console.log('errorLogin 2 ==>', error)
        if (error instanceof BadCredentialsException) {
          console.log('errorLogin 3 ==>', error.message)
          Util.showWarnToast(toast, error.message);
        } else {
          console.log('errorLogin 4 ==>', error)
          Util.showErrorToast(toast, t("errorLogin"));
        }
      }
    };

    const validationSchema = Yup.object({
      email: Yup.string().email(t("invalidEmail")).required(t("requiredField")),
      password: Yup.string().required(t("requiredField")),
    });

    return (
      <>
      {/* @ts-ignore */}
        <Toast ref={toast} position="top-center" />
        {/* @ts-ignore */}
        <Dialog
          header={t(forgotPassword ? "emailToRecover" : loginActive ? "emailAndPassword" : "registerOrLogin")}
          visible={authModal}
          onHide={() => onHide()}
        >
          <ContentWrapper>
            {loginActive ? (
              <Formik
                initialValues={initialValues}
                validationSchema={validationSchema}
                onSubmit={onSubmit}
                validateOnChange
              >
                {(formik) => {
                  return (
                    <Form>
                      <Input
                        id="email"
                        name="email"
                        label={t("email")}
                        placeholder="example@gmail.com"
                        error={formik.errors.email && formik.touched.email}
                        icon={<Mail width={20} height={20}></Mail>}
                      />
                      {!forgotPassword && (
                        <Input
                          id="password"
                          name="password"
                          label={t("password")}
                          type="password"
                          placeholder="******"
                          error={formik.errors.password && formik.touched.password}
                          icon={<Password width={20} height={20}></Password>}
                        />
                      )}

                      <ButtonContainer>
                        {!forgotPassword ? (
                          <>
                            <span>
                              <Button
                                text={t("login")}
                                disabled={!formik.isValid}
                                onClick={() => setLoginActive(true)}
                              ></Button>
                            </span>
                            <span>
                              <Button
                                type="button"
                                text={t("createAccount")}
                                color={theme.secondaryBackground}
                                onClick={createAccount}
                              ></Button>
                            </span>
                          </>
                        ) : (
                          <>
                            <Button
                              type="button"
                              text={t("recoverPassword")}
                              disabled={formik.errors.email ? true : false}
                              onClick={() => recoverPassword(formik.values.email)}
                            ></Button>
                          </>
                        )}
                      </ButtonContainer>

                      <ForgotPassword>
                        <Text onClick={() => setForgotPassword(!forgotPassword)}>
                          {t(forgotPassword ? "login" : "forgotPassword")}
                        </Text>
                      </ForgotPassword>
                    </Form>
                  );
                }}
              </Formik>
            ) : (
              <>
                <p>{t("registerOrLoginToAcess")}</p>
                <ButtonContainer>
                  <span>
                    <Button text={t("login")} onClick={() => setLoginActive(true)}></Button>
                  </span>
                  <span>
                    <Button
                      text={t("register")}
                      color={theme.secondaryBackground}
                      onClick={() => router.push("/register")}
                    ></Button>
                  </span>
                </ButtonContainer>
              </>
            )}
          </ContentWrapper>
        </Dialog>
{/* @ts-ignore */}
        <Component setAuthModalOnLogin={setAuthModalOnLogin} setAuthModal={setAuthModal} {...props} />
      </>
    );
  };

  return Wrapper;
}
