import styled, { useTheme } from 'styled-components';
import Link from 'next/link';
import Heart from '../SvgIcons/heart';
import WithAuthModal from '../WithAuthModal';
import { Dispatch, SetStateAction, useEffect, useRef, useState } from 'react';
import { Sidebar } from 'primereact/sidebar';
import {
  IoPersonOutline,
  IoHomeOutline,
  IoDocumentTextOutline,
  IoShareSocialSharp,
} from 'react-icons/io5';
import { RiLogoutBoxLine, RiLoginBoxLine, RiRoadMapLine, RiTeamLine, RiLineChartFill } from 'react-icons/ri';
import { TieredMenu } from 'primereact/tieredmenu';
import { InputSwitch } from 'primereact/inputswitch';

import { AiOutlineMenuFold } from 'react-icons/ai';
import router, { Router } from 'next/router';
import { useUserContext } from '../../../core/contexts/UserContext';
import useAuth from '../../../core/hooks/useAuth';
import { MenuItem, MenuItemOptions } from 'primereact/menuitem';
import useUser from '../../../core/hooks/useUser';
import {
  languages,
  usePreferencesContext,
} from '../../../core/contexts/PreferencesContext';
import { Util } from '../../../core/utils/util';
import Button from '../Button';
import { useTranslation } from 'next-i18next';
import Adventures from '../SvgIcons/adventure';
import { FaUsers, FaRegFileAlt, FaClipboardList } from 'react-icons/fa'

interface NavBarProps {
  setAuthModal: Dispatch<SetStateAction<boolean>>;
  setAuthModalOnLogin: () => void;
}

function NavBar({ setAuthModalOnLogin }: NavBarProps) {
  const [showSideBar, setShowSideBar] = useState(false);
  const theme = useTheme();
  const { user, isAuthenticated } = useUserContext();
  const { changeLanguage, preferences } = usePreferencesContext();
  const [isMobile, setIsMobile] = useState(false);
  const { t } = useTranslation('common');

  const menu = useRef<any>();
  const menuLanguage = useRef<any>();

  const { logout } = useAuth();
  const { alterProfile } = useUser();
  const [isSupplier, setIsSupplier] = useState(
    user?.perfilAtual == 'E' ? true : false
  );
  const [isAdmin, setIsAdmin] = useState(false)

  const hideSideBar = () => {
    setShowSideBar(false);
  };

  const hideMenu = () => {
    menu.current?.hide();
  };

  const getClassActive = (href: string): string => {
    try {
      return href.includes(window.location.pathname) ? 'li-active' : '';
    } catch (error) {
      return '';
    }
  };

  const SidebarMobileItem = ({ href = '/', icon, text, needAuth }: any) => {
    if (needAuth && !isAuthenticated) {
      return <></>;
    }
    return (
      <li className={getClassActive(href)}>
        <Link href={href} passHref>
          <a>
            {icon}
            <MobileItemName> {text}</MobileItemName>
          </a>
        </Link>
      </li>
    );
  };

  const SidebarMobileItemWithoutLink = ({
    icon,
    text,
    needAuth,
    onClick,
    withSwitch = false,
  }: any) => {
    if (needAuth && !isAuthenticated) {
      return <></>;
    }
    return (
      <li onClick={onClick}>
        <a className={' mini-switch'}>
          {icon}
          <MobileItemName> {text}</MobileItemName>
          {/* @ts-ignore */}
          {withSwitch && (
            <InputSwitch
              checked={isSupplier}
              onChange={(e: any) => {
                handleAlterProfile(!isSupplier);
              }}
            />
          )}
        </a>
      </li>
    );
  };

  const SidebarMobileItemWithoutLinkAdmin = ({ icon, text, needAuth, onClick, withSwitch = false }: any) => {
    if (needAuth && !isAuthenticated) {
      return <></>;
    }
    return (
      <li onClick={onClick}>
        <a className={" mini-switch"}>
          {icon}
          <text> {text}</text>
          {withSwitch && (
            <InputSwitch
              checked={isAdmin}
              onChange={(e: any) => {
                handleAlterProfileAdmin(!isAdmin);
              }}
            />
          )}
        </a>
      </li>
    );
  };

  const SidebarItem = ({ href = "", icon, text, needAuth }: any) => {
    if (needAuth && !isAuthenticated) {
      return <></>;
    }
    return (
      <li>
        <Link href={href}>
          <a>
            {icon}
            <text> {text}</text>
          </a>
        </Link>
      </li>
    );
  };

  const SidebarItemWithoutLink = ({ icon, text, needAuth, onClick }: any) => {
    if (needAuth && !isAuthenticated) {
      return <></>;
    }
    return (
      <li onClick={onClick}>
        <a>
          {icon}
          <text> {text}</text>
        </a>
      </li>
    );
  };

  const CustomMenuItem = (
    item: MenuItem,
    options: MenuItemOptions,
    withSwitch = false,
    url?: string
  ) => {
    return (
      <a
        className={options.className + ' mini-switch'}
        style={item.style}
        target={item.target}
        onClick={options.onClick}
      >
        <span className={options.iconClassName} style={item.style}>
          {url && <LanguageIcon url={url} />}
        </span>
        <span className={options.labelClassName} style={item.style}>
          {item.label}
        </span>

        {withSwitch && (
          <InputSwitch
            checked={isSupplier}
            onChange={(e: any) => {
              handleAlterProfile(!isSupplier);
            }}
          />
        )}
      </a>
    );
  };

  const CustomMenuItemAdmin = (item: MenuItem, options: MenuItemOptions, withSwitch = false, url?: string) => {
    return (
      <a
        className={options.className + " mini-switch"}
        style={item.style}
        target={item.target}
        onClick={options.onClick}
      >
      <span className={options.iconClassName} style={item.style}>
        {url && <LanguageIcon url={url} />}
      </span>
      <span className={options.labelClassName} style={item.style}>
        {item.label}
      </span>

      {withSwitch && (
        <InputSwitch
          checked={isAdmin}
          onChange={(e: any) => {
            handleAlterProfileAdmin(!isAdmin);
          }}
        />
      )}
      </a>
    );
  }

  const getImageUrlByProfile = () => {
    let imageUrl = "/assets/perfil-default.png";
    if (user.perfilAtual == "E" && user.urlFotoEstabelecimento) {
      imageUrl = user.urlFotoEstabelecimento;
    }
    if (user.perfilAtual == "C" && user.urlFotoCliente) {
      imageUrl = user.urlFotoCliente;
    }

    if(!isSupplier && isAdmin){
      imageUrl = "/assets/perfil-default.png";
    }

    return imageUrl;
  };

  const getImageByLanguage = (): string => {
    if (preferences.language === 'en') {
      return '/assets/english.png';
    } else if (preferences.language === 'es') {
      return '/assets/spain.png';
    }

    return '/assets/brazil.png';
  };

  const handleAlterProfile = async (isSupplier: boolean) => {
    await alterProfile(isSupplier ? 'E' : 'C');
    setIsSupplier(isSupplier);
    setIsAdmin(false)
  };
  
  const handleAlterProfileAdmin = async (isAdmin: boolean) => {
    setIsAdmin(isAdmin)
    setIsSupplier(false)
  }

  const openTerms = () => {
    window.open('https://reltyapp.com.br/termos-usuario');
    hideMenu();
  };

  const sendToFriends = async () => {
    if (isMobile) {
      const shareData = {
        title: t("downloadRelty"),
        text: t("downloadRelty") + " " + "https://reltyapp.com.br",
      };
      await navigator.share(shareData);
    } else {
      window.open(
        'https://api.whatsapp.com/send?text=' +
          t('downloadRelty') +
          ' ' +
          'https://reltyapp.com.br',
        '_blank'
      );
    }

    hideMenu();
  };

  const itemsRender = (): MenuItem[] => {
    if(user.seAdministrador == 'S'){
      return [{
        label: t("editProfileMenu"),
        icon: "pi pi-fw pi-pencil",
        command: () => {
          router.push(`/profiles/${isSupplier ? "supplier" : "user"}`);
        },
      },
      {
        icon: "pi pi-fw pi pi-home",
        label: t("supplier"),
        template: (item, options) => CustomMenuItem(item, options, true),
      },
      {
        icon: "pi pi-fw pi pi-user",
        label: "Admin",
        template: (item, options) => CustomMenuItemAdmin(item, options, true),
      },
      {
        label: t("recommendToAFriend"),
        icon: "pi pi-fw pi-user",
        command: sendToFriends,
      },
      {
        label: t("termsOfService"),
        icon: "pi pi-fw pi-file",
        command: openTerms,
      },
      {
        separator: true,
      },
      {
        icon: "pi pi-fw pi-power-off",
        label: t("logout"),
        style: { color: theme.red },
        command: logout,
        template: (item, options) => CustomMenuItem(item, options),
      }]
    } else {
      return [{
        label: t("editProfileMenu"),
        icon: "pi pi-fw pi-pencil",
        command: () => {
          router.push(`/profiles/${isSupplier ? "supplier" : "user"}`);
        },
      },
      {
        icon: "pi pi-fw pi pi-home",
        label: t("supplier"),
        template: (item, options) => CustomMenuItem(item, options, true),
      },
      {
        label: t("recommendToAFriend"),
        icon: "pi pi-fw pi-user",
        command: sendToFriends,
      },
      {
        label: t("termsOfService"),
        icon: "pi pi-fw pi-file",
        command: openTerms,
      },
      {
        separator: true,
      },
      {
        icon: "pi pi-fw pi-power-off",
        label: t("logout"),
        style: { color: theme.red },
        command: logout,
        template: (item, options) => CustomMenuItem(item, options),
      }]
    }
  }

  const changeLanguageAndReload = (language: languages) => {
    changeLanguage(language);
  };

  const handleWantToOfferButton = () => {
    if (Util.isMobile()) {
      setTimeout(function () {
        if (Util.isMobile()) {
          window.location.href = 'https://relty.app.link/';
        }
        return;
      }, 800);
      window.location.href = `relty://?cadastrarAventura=true`;
    } else {
      window.location.href = '/profiles/supplier/adventures/create';
    }
  };

  const languageItems: MenuItem[] = [
    {
      label: 'English - EN',
      command: () => changeLanguageAndReload('en'),
      template: (item, options) =>
        CustomMenuItem(item, options, false, '/assets/english.png'),
    },
    {
      label: 'Español - ES',
      command: () => changeLanguageAndReload('es'),
      template: (item, options) =>
        CustomMenuItem(item, options, false, '/assets/spain.png'),
    },
    {
      label: 'Português - PT',
      command: () => changeLanguageAndReload('pt'),
      template: (item, options) =>
        CustomMenuItem(item, options, false, '/assets/brazil.png'),
    },
  ];

  useEffect(() => {
    Router.events.on('routeChangeStart', hideSideBar);
    Router.events.on('routeChangeStart', hideMenu);

    setIsMobile(Util.isMobile());
  }, []);

  useEffect(() => {
    setIsSupplier(user?.perfilAtual == 'E' ? true : false);
  }, [user]);

  return (
    <nav
      style={{
        height: '50%',
      }}
    >
      <Wrapper>
        <NavBarDesktop>

          {/* =========== Authenticated and Supplier ============== */}
          {isSupplier && isAuthenticated && (
            <>
              <Button
                onClick={handleWantToOfferButton}
                text={t('wantToOfferAnAdventure')}
              ></Button>

              <SidebarItem
                href="/profiles/supplier/overview"
                text={t('overview')}
                icon={
                  <RiLineChartFill size={26} color={theme.primaryBackground} />
                }
              />

              <SidebarItem
                href="/profiles/supplier/adventures"
                text={t('adventures')}
                icon={
                  <Adventures
                    width={24}
                    height={26}
                    fill={theme.primaryBackground}
                 />
                }
              />

              <SidebarItem
                href="/profiles/supplier/clients"
                text={t('customers')}
                icon={<RiTeamLine size={26} color={theme.primaryBackground} />}
              />
            </>
          )}

          {!isAuthenticated && (
            <>
              <SidebarItemWithoutLink
                text={t('login')}
                icon={
                  <RiLoginBoxLine size={26} color={theme.primaryBackground} />
                }
                onClick={() => setAuthModalOnLogin()}
              />
              <SidebarItem
                href="/register"
                text={t('register')}
                icon={
                  <IoPersonOutline size={26} color={theme.primaryBackground} />
                }
              />
            </>
          )}

          {/* ========================== user admin ==================== */}
          {isAuthenticated && isAdmin && !isSupplier &&(
            <>
              <SidebarItem
                href="/profiles/admin/overview"
                text={t("overview")}
                icon={<RiLineChartFill size={26} color={theme.primaryBackground}/>}
              />

              <SidebarItem
                href="/profiles/admin/supplier"
                text="Fornecedores"
                needAuth
                icon={<FaUsers size={26} fill={theme.primaryBackground} />}
              />

              <SidebarItem
                href="/profiles/admin/user"
                text="Usuários"
                needAuth
                icon={<FaUsers size={26} fill={theme.primaryBackground} />}
              />

              <SidebarItem
                href="/profiles/admin/materials"
                text="Materiais"
                needAuth
                icon={<FaRegFileAlt size={26} fill={theme.primaryBackground} />}
              />

              <SidebarItem
                href="/profiles/admin/categories"
                text="Categorias"
                needAuth
                icon={<FaClipboardList size={26} fill={theme.primaryBackground} />}
              />

              {/* <SidebarItem
                href="/favorites?page=1"
                text={"_Aventuras"}
                needAuth
                icon={<RiTeamLine size={26} color={theme.primaryBackground} />}
              /> */}
            </>
          )}

          {/* ===================== user normal =================== */}

          {isAuthenticated && !isSupplier && !isAdmin &&(
            <>
              <SidebarItem
                href="/map"
                text={t("exploreInMap")}
                needAuth
                icon={<RiRoadMapLine size={26} color={theme.primaryBackground} />}
                />

              <SidebarItem
                href="/favorites?page=1"
                text={t("favorites")}
                needAuth
                icon={<Heart width={24} height={26} fill={theme.primaryBackground} />}
              />

              <SidebarItem
                href="/adventures?page=1"
                text={"Aventuras"}
                needAuth
                icon={<Adventures width={24} height={26} color={theme.primaryBackground} />}
              />

              {/* <SidebarItem
                href="/community"
                text={t("community")}
                needAuth
                icon={<RiTeamLine size={26} color={theme.primaryBackground} />}
              /> */}
            </>
          )}

        {isAuthenticated && (
          <>
            <li>
              <a>
                {/* @ts-ignore */}
                <TieredMenu model={itemsRender()} popup ref={menu} id="overlay_tmenu" />
                <ImageRadius url={getImageUrlByProfile()} onClick={(event: any) => menu.current?.toggle(event)} />
              </a>
            </li>
          </>
        )}

        <>
          <li>
            <a>
              {/* @ts-ignore */}
              <TieredMenu model={languageItems} popup ref={menuLanguage} id="overlay_tmenu" />
              <ImageRadius
                url={getImageByLanguage()}
                onClick={(event: any) => {
                  menuLanguage.current?.toggle(event);
                }}
              />
            </a>
          </li>
        </>
        </NavBarDesktop>

        <NavBarMobile>
          {isSupplier && isAuthenticated && (
            <AdventureIconWrapper>
              <Adventures onClick={handleWantToOfferButton}></Adventures>
              <text>{t('newAdventure')}</text>
            </AdventureIconWrapper>
          )}

          <>
            <li>
              <a>
                {/* @ts-ignore */}
                <TieredMenu
                  model={languageItems}
                  popup
                  ref={menuLanguage}
                  id="overlay_tmenu"
                />
                <ImageRadius
                  url={getImageByLanguage()}
                  onClick={(event: any) => menuLanguage.current?.toggle(event)}
                />
              </a>
            </li>
          </>

          <AiOutlineMenuFold size={35} color={theme.pageForeground} onClick={() => setShowSideBar(true)} />
          {/* @ts-ignore */}
          <Sidebar visible={showSideBar} position="right" onHide={() => setShowSideBar(false)}>
            <ContainerImage>
              <ImageRadius url={getImageUrlByProfile()} />
            </ContainerImage>

            <SidebarMobileItem
              href="/home?page=1"
              text={t('home')}
              icon={<IoHomeOutline size={24} color={theme.primaryBackground} />}
              />

            {isAuthenticated && (
              
              <SidebarMobileItemWithoutLink
                text={t('supplier')}
                icon={
                  <IoHomeOutline size={24} color={theme.primaryBackground} />
                }
                withSwitch
              />
            )}

            {isAuthenticated && user.seAdministrador == "S" && (
              <SidebarMobileItemWithoutLinkAdmin
                text="Admin"
                icon={<IoHomeOutline size={24} color={theme.primaryBackground} />}
                withSwitch
              />
            )}
            
            {/* =========== Authenticated and Supplier ============== */}
            {isSupplier && isAuthenticated && !isAdmin && (
            <>
              <SidebarMobileItem
                href="/profiles/supplier/overview"
                text={t("overview")}
                icon={<RiLineChartFill size={26} color={theme.primaryBackground}/>}
              />

              <SidebarMobileItem
                href="/profiles/supplier/adventures"
                text={t("adventures")}
                icon={<Adventures width={24} height={26} fill={theme.primaryBackground}/>}
              />

              <SidebarMobileItem
                href="/profiles/supplier/clients"
                text={t("customers")}
                icon={<RiTeamLine size={26} color={theme.primaryBackground}/>}
              />
            </>
            )}

            {/* ======================= user admin ===================== */}

            {isAuthenticated && !isSupplier && isAdmin && (
              <>
                <SidebarMobileItem
                  href="/profiles/admin/overview"
                  text="Overview"
                  icon={<RiRoadMapLine size={26} color={theme.primaryBackground} />}
                  needAuth
                />

                <SidebarMobileItem
                  href="/profiles/admin/supplier"
                  text="Fornecedores"
                  needAuth
                  icon={<FaUsers size={26} fill={theme.primaryBackground} />}
                />

                <SidebarMobileItem
                  href="/profiles/admin/user"
                  text="Usuários"
                  needAuth
                  icon={<FaUsers size={26} fill={theme.primaryBackground} />}
                />

                <SidebarMobileItem
                  href="/profiles/admin/materials"
                  text="Materiais"
                  needAuth
                  icon={<FaRegFileAlt size={26} fill={theme.primaryBackground} />}
                />

              <SidebarMobileItem
                href="/profiles/admin/categories"
                text="Categorias"
                needAuth
                icon={<FaClipboardList size={26} fill={theme.primaryBackground} />}
              />

                <SidebarMobileItem
                  href="/favorites?page=1"
                  text={"_Aventuras"}
                  needAuth
                  icon={<RiTeamLine size={26} color={theme.primaryBackground} />}
                />
              </>
            )}

            {/* ===================== user normal =================== */}
            {isAuthenticated && !isSupplier && !isAdmin && (
              <>
                <SidebarMobileItem
                  href="/map"
                  text={t("exploreInMap")}
                  icon={<RiRoadMapLine size={26} color={theme.primaryBackground} />}
                />

                <SidebarMobileItem
                  href="/favorites?page=1"
                  text={t("favorites")}
                  needAuth
                  icon={<Heart width={24} height={26} fill={theme.primaryBackground} />}
                />

              <SidebarMobileItem
                  href="/adventures?page=1"
                  text={"Aventuras"}
                  needAuth
                  icon={<Adventures width={24} height={26} color={theme.primaryBackground} />}
              />

              <SidebarMobileItem
                  href="/community"
                  text={t("community")}
                  needAuth
                  icon={<RiTeamLine size={26} color={theme.primaryBackground} />}
              />
              </>
            )}

            {!isAuthenticated && (
              <>
                <SidebarMobileItemWithoutLink
                  text={t('login')}
                  icon={
                    <RiLoginBoxLine size={26} color={theme.primaryBackground} />
                  }
                  onClick={() => {
                    setAuthModalOnLogin();
                    setShowSideBar(false);
                  }}
                />
                <SidebarMobileItem
                  href="/register"
                  text={t('register')}
                  icon={
                    <IoPersonOutline
                      size={26}
                      color={theme.primaryBackground}
                    />
                  }
                />
              </>
            )}

            <SidebarMobileItem
              href="/favorites?page=1"
              text={t('favorites')}
              needAuth
              icon={
                <Heart width={24} height={26} fill={theme.primaryBackground} />
              }
            />

            <SidebarMobileItem
              href={`/profiles/${isSupplier ? 'supplier' : 'user'}`}
              text={t('editProfileMenu')}
              needAuth
              icon={
                <IoPersonOutline size={24} color={theme.primaryBackground} />
              }
            />

            <SidebarMobileItemWithoutLink
              onClick={openTerms}
              text={t('termsOfService')}
              icon={
                <IoDocumentTextOutline
                  size={24}
                  color={theme.primaryBackground}
                />
              }
            />

            <SidebarMobileItemWithoutLink
              onClick={sendToFriends}
              text={t('recommendToAFriend')}
              icon={
                <IoShareSocialSharp size={24} color={theme.primaryBackground} />
              }
            />

            {isAuthenticated && (
              <LogoutContainer
                className={'logout-item'}
                onClick={() => logout()}
              >
                <li>
                  <a>
                    <RiLogoutBoxLine size={24} color={theme.red} />
                    <text>{t('logout')} </text>
                  </a>
                </li>
              </LogoutContainer>
            )}
          </Sidebar>
        </NavBarMobile>
      </Wrapper>
    </nav>
  );
}

export default WithAuthModal(NavBar);

const Wrapper = styled.ul`
  display: flex;
  list-style: none;
  justify-content: center;
  align-items: center;
  cursor: pointer;

  a {
    color: ${(p) => p.theme.pageForeground};
    text-decoration: none;
    text-transform: capitalize;
    display: flex;
    flex-direction: column;
    justify-content: center;
    align-items: center;
    margin: 0 5px 0 5px;

    text {
      text-align: center;
      font: normal normal 600 9px/13px Poppins;
      letter-spacing: 0px;
      color: #006993;
      opacity: 1;
    }
    &:hover {
      color: ${(p) => p.theme.primaryBackground};
    }
  }
`;

const NavBarDesktop = styled.div`
  height: 100%;
  display: flex;
  justify-content: center;
  align-items: center;

  @media screen and (max-width: 767px) {
    display: none;
  }

  li {
    height: 100%;
    min-width: 64px;
    position: relative;
    align-self: flex-start;
    text {
      width: 36;
      position: absolute;
      bottom: -20px;
    }
  }
`;

const ImageRadius = styled.div<{ url: string }>`
  width: 40px;
  height: 40px;
  background-image: url(${(p) => p.url});
  background-position: center;
  background-size: cover;
  border-radius: 100px;
  cursor: pointer;
`;

const LanguageIcon = styled.div<{ url: string }>`
  width: 24px;
  height: 24px;
  background-image: url(${(p) => p.url});
  background-position: center;
  background-size: cover;
  border-radius: 100px;
  cursor: pointer;
`;

const ContainerImage = styled.div`
  width: 100%;
  display: flex;
  justify-content: center;
  align-items: center;
  margin-bottom: 10px;
`;

const MobileItemName = styled.text`
  color: red;
  margin-left: 5px;
`;

const NavBarMobile = styled.div`
  display: none;
  cursor: pointer;

  @media screen and (max-width: 767px) {
    display: flex;
    justify-content: center;
    align-items: center;
  }

  :active {
    opacity: 0.5;
  }
`;

const AdventureIconWrapper = styled.div`
  display: flex;
  flex-direction: column;
  justify-content: center;
  align-items: center;
  margin-right: 15px;
  position: relative;

  text {
    min-width: 60px;
    width: 50px;
    font-size: 8px;
    position: absolute;
    text-align: center;
    bottom: -13px;
    color: ${(p) => p.theme.primaryBackground};
  }
`;

const LogoutContainer = styled.div``;