import React, { useState } from "react";
import styled from "styled-components";
import { AdventureListDto } from "../../../core/models/interfaces/Adventure/AdventureListDto";

import { FormattedNumber } from "react-intl";
import { IntlProvider } from "react-intl";
import { light } from "../../../styles/theme";
import { Util } from "../../../core/utils/util";
import Star from "../SvgIcons/star";

export interface Translations {
  adventureType: string;
  where: string;
  perPerson: string;
}
interface MapCardProps {
  adventure: AdventureListDto;
  t: Translations;
}

function MapCard(props: MapCardProps) {
  const { adventure, t } = props;

  return (
    <IntlProvider messages={require("../../../../public/locales/pt/common.json")} locale="pt-BR" defaultLocale="pt-BR">
      <Wrapper id={`adventure-${adventure.id}`}>
        <AdventureImage url={adventure.urlFoto}>
          <Tag backgroudColor="#FFF">
            <ElementCircle color={Util.getColorByAdventureType(adventure.tipo)}></ElementCircle>
            <text>{t.adventureType}</text>
          </Tag>

          {adventure?.avaliacao && !isNaN(adventure?.avaliacao) && (
            <RatingContainer>
              <Star />
              <Text>{parseFloat(props.adventure?.avaliacao?.toString()).toFixed(1)}</Text>
            </RatingContainer>
          )}
        </AdventureImage>
        <InfoContainer>
          <Title>{props.adventure?.titulo}</Title>
          <Text>
            {t.where}: {adventure?.local}
          </Text>
        </InfoContainer>

        <PriceContainer>
          <Title>
            <FormattedNumber value={adventure?.valor} style="currency" currency="BRL" />
          </Title>
          <Text>{t.perPerson}</Text>
        </PriceContainer>
      </Wrapper>
    </IntlProvider>
  );
}

export default MapCard;

const Wrapper = styled.div`
  width: 100%;
  min-width: 250px;
  max-width: 250px;
  min-height: 250px;
  cursor: pointer;
  position: relative;
`;

export const InfoWrapper = styled.div``;

export const InfoContainer = styled.div`
  position: absolute;
  top: 69%;
  width: 65%;
  height: 30%;
  padding: 10px 10px;
  display: flex;
  flex-direction: column;
`;

export const Title = styled.text`
  text-align: left;
  font: normal normal 600 12px Poppins;
  letter-spacing: 0.2px;
  line-height: 1.1;
  color: ${light.pageForeground};
`;

export const Text = styled(Title)`
  font: normal normal normal 11px Poppins;
  margin-top: 10px;
`;

export const PriceContainer = styled.div`
  position: absolute;
  top: 69%;
  width: 35%;
  height: 30%;
  padding: 10px 10px;
  display: flex;
  flex-direction: column;
  right: 0;
  padding-left: 0;
`;

export const Tag = styled.div<{ backgroudColor?: string }>`
  right: 8px;
  top: 142px;
  position: absolute;
  display: flex;
  z-index: 10;
  width: 60px;
  padding: 5px 5px;
  height: 20px;
  flex-direction: row;
  justify-content: space-around;
  align-items: center;
  background-color: ${light.activeElementBackground};
  border-radius: ${light.borderRadius};
  color: ${light.pageForeground};
  position: absolute;
  right: 8px;

  text {
    font: normal normal normal 12px/14px Poppins;
    margin-left: 4px;
  }
`;

export const RatingContainer = styled(Tag)`
  left: 8px;
  top: 8px;
  max-width: 55px;
  height: 22px;
  background-color: ${(p) => p.theme.activeElementBackground};
  border-radius: ${(p) => p.theme.borderRadius};
  padding: 5px;
  justify-content: space-around;
`;

export const ElementCircle = styled.div<{ color: string }>`
  width: 14px;
  height: 14px;
  border-radius: ${light.borderRadius};
  background-color: ${(p) => p.color};
  position: absolute;
  left: 2px;
`;

export const AdventureImage = styled.div<{ url: string }>`
  position: absolute;
  top: 0;
  left: 0;
  width: 100%;
  height: 69%;
  background-image: url(${(p) => p.url});
  background-position: center;
  background-size: cover;
  border-top-left-radius: ${(p) => p.theme.borderRadius};
  border-top-right-radius: ${(p) => p.theme.borderRadius};
`;
