import React, { useCallback, useState } from "react";
import { InputText } from "primereact/inputtext";
import { useTheme } from "styled-components";
import { AdventureType } from "../../../core/models/enums/AdventureType";
import All from "../SvgIcons/all";

import {
  Wrapper,
  Elements,
  ElementContainer,
  ElementName,
  InputWrapper,
  ButtonContainer,
  SimpleFilters,
  FilterTitle,
  FiltersContainer,
  InputContainer,
  AutoComplete,
} from "./styles";
import Button from "../Button";
import Earth from "../SvgIcons/earth";
import Air from "../SvgIcons/air";
import Water from "../SvgIcons/water";
import { useTranslation } from "next-i18next";
import { IoIosArrowDown } from "react-icons/io";
import { MdOutlineAttachMoney } from "react-icons/md";
import { FaSearch } from "react-icons/fa";
import { RiPinDistanceFill, RiSortAsc } from "react-icons/ri";

import { Collapse } from "react-collapse";
import { AdvancedFilters } from "../../../core/models/interfaces/Adventure/AdvancedFilters";
import InputFilter from "../InputFilter";
import InputMoney from "../InputMoney";
import SelectFilter from "../SelectFilter";
import { Option } from "react-dropdown";
import _ from "lodash";
import useMap from "../../../core/hooks/useMap";

interface HomeSearchBarProps {
  activeElement?: AdventureType | null;
  setActiveElement: (newElement: AdventureType | null | undefined) => void;
  search?: string | string[] | null;
  setSearch: (text: string | string[] | null | undefined) => void;
  handleSearchAventure: (event?: React.KeyboardEvent<HTMLInputElement>) => void;
  advancedFilters?: AdvancedFilters;
  clearFilters?: () => void;
  reposisitionMap?: (placeId: string) => void;
}

function HomeSearchBar(props: HomeSearchBarProps) {
  const theme = useTheme();
  const {
    handleSearchAventure,
    setActiveElement,
    activeElement,
    setSearch,
    search,
    advancedFilters,
    clearFilters,
    reposisitionMap,
  } = props;
  const [filterAdvanced, setFilterAdvanced] = useState(false);

  const { searchPlaces } = useMap();

  const [autoCompleteItems, setAutoCompleteItems] = useState<google.maps.places.AutocompletePrediction[]>([]);
  const [selectedLocation, setSelectedLocation] = useState<string | null>(null);
  const [autoCompleteVisible, setAutoCompleteVisible] = useState<boolean>(false);
  const [loading, setLoading] = useState<boolean>(false);

  const { t } = useTranslation("common");

  //AdvancedFilters
  const changePrice = (event: React.ChangeEvent<HTMLInputElement>) => {
    advancedFilters?.price?.setMethod(event.target.value);
  };

  const changeOrder = (option: Option) => {
    advancedFilters?.order?.setMethod(option.value);
  };

  const changeDistance = (event: React.ChangeEvent<HTMLInputElement>) => {
    advancedFilters?.distance?.setMethod(event.target.value);
  };

  const changeLocation = (event: React.ChangeEvent<HTMLInputElement>) => {
    advancedFilters?.location?.setMethod(event.target.value);
    searchDebounce(event.target.value);
  };

  const selectLocation = async (prediction: google.maps.places.AutocompletePrediction) => {
    setSelectedLocation(prediction.place_id);
    if (reposisitionMap) reposisitionMap(prediction.place_id);
  };

  const searchCities = (text: string) => {
    if (!text || text.length < 1) {
      setAutoCompleteVisible(false);
    }

    if (text && !loading) {
      setAutoCompleteVisible(true);
      setLoading(true);
      searchPlaces(text)
        .then((result) => {
          if (result) {
            setAutoCompleteItems(result);
          }
        })
        .finally(() => {
          setLoading(false);
        });
    } else {
      setAutoCompleteItems([]);
    }
  };

  const searchDebounce = useCallback(
    _.debounce((text: string) => searchCities(text), 300),
    [autoCompleteItems]
  );

  const cleanMapFilters = () => {
    if (clearFilters) {
      clearFilters();
    }

    setAutoCompleteItems([]);
    setSelectedLocation(null);
    setAutoCompleteVisible(false);
  };

  const orderOptions: Option[] = [
    { value: "", label: t("selectAnItem") },
    { value: "M", label: t("mostPracticed") },
  ];

  const getColor = (type: string | null): string => {
    if (type == activeElement) {
      return theme.primaryBackground;
    } else {
      return theme.inactiveElementForeground;
    }
  };

  return (
    <Wrapper>
      <SimpleFilters>
        <Elements>
          <ElementContainer onClick={() => setActiveElement(AdventureType.T)}>
            <Earth width={55} height={55} active={activeElement === AdventureType.T ? true : false} />
            <ElementName color={getColor("T")}>{t("earth")}</ElementName>
          </ElementContainer>

          <ElementContainer onClick={() => setActiveElement(AdventureType.G)}>
            <Water width={55} height={55} active={activeElement === AdventureType.G ? true : false} />
            <ElementName color={getColor("G")}>{t("water")}</ElementName>
          </ElementContainer>

          <ElementContainer onClick={() => setActiveElement(AdventureType.A)}>
            <Air width={55} height={55} active={activeElement === AdventureType.A ? true : false} />
            <ElementName color={getColor("A")}>{t("air")}</ElementName>
          </ElementContainer>

          <ElementContainer onClick={() => setActiveElement(null)}>
            <All width={55} height={55} active={activeElement === null ? true : false} />
            <ElementName color={getColor(null)}>{t("all")}</ElementName>
          </ElementContainer>
        </Elements>

        <InputWrapper>
          <span className="p-input-icon-left">
            <FaSearch size={19} color={theme.pageForeground} className="pi pi-search" />
            {/* @ts-ignore */}
            <InputText
              placeholder={t("searchYourAdventure")}
              className="p-inputtext-md"
              onChange={(event) => setSearch(event.target?.value)}
              value={search || ""}
              onKeyDown={(event) => handleSearchAventure(event)}
            />
          </span>
          <ButtonContainer>
            <Button onClick={() => handleSearchAventure()} text={t("search")}></Button>
          </ButtonContainer>
        </InputWrapper>
      </SimpleFilters>

      {advancedFilters && (
        <>
          <FilterTitle
            rotateAnimation={filterAdvanced}
            onClick={() => {
              setFilterAdvanced(!filterAdvanced);
            }}
          >
            {t("advancedFilter")}
            <IoIosArrowDown size={24} />
          </FilterTitle>

          <Collapse isOpened={filterAdvanced}>
            <>
              <FiltersContainer>
                {advancedFilters.price?.visible && (
                  <InputContainer>
                    <label>{t("maxPrice")}</label>
                    <span className="p-input-icon-left">
                      <MdOutlineAttachMoney size={19} color={theme.pageForeground} className="pi pi-search" />
                      <InputMoney
                        placeholder={t("maxPrice")}
                        className="p-inputtext-md"
                        onChange={changePrice}
                        value={advancedFilters?.price?.value || ""}
                        onKeyDown={(event) => handleSearchAventure(event)}
                      />
                    </span>
                  </InputContainer>
                )}

                {advancedFilters.order?.visible && (
                  <InputContainer>
                    <label>{t("orderBy")}</label>
                    <span className="p-input-icon-left">
                      <RiSortAsc size={19} color={theme.pageForeground} className="pi pi-search" />
                      <SelectFilter
                        placeholder={t("selectAnItem")}
                        className="p-inputtext-md"
                        onChange={changeOrder}
                        options={orderOptions}
                        value={advancedFilters?.order?.value}
                      />
                    </span>
                  </InputContainer>
                )}

                {advancedFilters.distance?.visible && (
                  <InputContainer>
                    <label>{t("distance")}</label>
                    <span className="p-input-icon-left">
                      <RiPinDistanceFill size={19} color={theme.pageForeground} className="pi pi-search" />
                      <InputFilter
                        placeholder={t("distance")}
                        mask="numbers"
                        className="p-inputtext-md"
                        onChange={changeDistance}
                        value={advancedFilters?.distance.value}
                        onKeyDown={(event) => handleSearchAventure(event)}
                      />
                    </span>
                  </InputContainer>
                )}

                {advancedFilters.location?.visible && (
                  <InputContainer>
                    <label>{t("location")}</label>
                    <span className="p-input-icon-left">
                      <RiPinDistanceFill size={19} color={theme.pageForeground} className="pi pi-search" />
                      <InputFilter
                        placeholder={t("São Paulo, Brasil")}
                        className="p-inputtext-md"
                        onChange={changeLocation}
                        value={advancedFilters?.location.value}
                        onKeyDown={(event) => handleSearchAventure(event)}
                        onBlur={() => {
                          setTimeout(() => {
                            setAutoCompleteVisible(false);
                          }, 300);
                        }}
                      />
                      <AutoComplete className="dropdown-filter" visible={autoCompleteVisible}>
                        {autoCompleteItems.map((prediction, index) => (
                          <div
                            key={index}
                            className={`Dropdown-option ${
                              selectedLocation == prediction.place_id ? "is-selected" : ""
                            }`}
                            role="option"
                            aria-selected="true"
                            onClick={() => selectLocation(prediction)}
                          >
                            {prediction.description}
                          </div>
                        ))}
                      </AutoComplete>
                    </span>
                  </InputContainer>
                )}
              </FiltersContainer>
              {clearFilters && (
                <FilterTitle
                  rotateAnimation={filterAdvanced}
                  onClick={() => {
                    cleanMapFilters();
                  }}
                >
                  {t("clearFilter")}
                </FilterTitle>
              )}
            </>
          </Collapse>
        </>
      )}
    </Wrapper>
  );
}

export default HomeSearchBar;
