import { Form, Formik } from 'formik';
import { useTranslation } from 'next-i18next';
import { Dialog } from 'primereact/dialog';
import { Toast } from 'primereact/toast';
import useFile from "../../../core/hooks/useFile";
import { ChangeEvent, useEffect, useRef, useState } from 'react';
import * as Yup from 'yup';
import YupPassword from "yup-password";
import "yup-phone";
import useUser from '../../../core/hooks/useUser';
import { Util } from '../../../core/utils/util';
import Button from '../Button';
import Input from '../CustomInput';
import Password from '../SvgIcons/password';
import { Wrapper, ButtonContainer, RegisterImage } from './styles';
import Cellphone from "../../components/SvgIcons/cellphone";
import Mail from "../../components/SvgIcons/mail";
import UserIcon from "../../components/SvgIcons/user";
import { masks } from "../../../core/utils/masks";
import { ImageError } from '../../../core/utils/exceptions';
import { InputSwitch } from 'primereact/inputswitch';
import { UserPermission } from '../../../core/models/interfaces/User/UserPermission';
import { UserRegisterDtoAdmin } from '../../../core/models/interfaces/User/UserRegisterDtoAdmin';

YupPassword(Yup);

const defaultPermission: UserPermission = {
    overview: false,
    suppliers: false,
    users: false,
    adventures: false,
    materials: false,
    categorys: false  
};

export default function CreateUsermodal(props: any) {
    const inputRef = useRef(null);
    const toast = useRef(null);
    const { getArchiveByHtmlInput, archiveInfos } = useFile("image");
    const [visible, setVisible] = useState(false)
    const [userPermission, setUserPermission] = useState<any>(defaultPermission)
    const { createAdminUser } = useUser()
    const { t } = useTranslation("common")

    useEffect(() => {
        setVisible(props.visible)
        setUserPermission(defaultPermission)
    }, [props.visible])

    const handleOnImageChange = (event: ChangeEvent<HTMLInputElement>) => {
        try {
            getArchiveByHtmlInput(event);
        } catch (error) {
            if (error instanceof ImageError)
                Util.showErrorToast(toast, error.message);
            else
                Util.showErrorToast(toast, t("onUploadImageError"));
        }
    };

    const onSubmit = async (values: any) => {
        let body: UserRegisterDtoAdmin = {
            foto: archiveInfos?.base64
                ? {
                bytes: archiveInfos?.base64,
                nome: archiveInfos?.nome as string,
                contentType: archiveInfos?.contentType as string,
                }
                : null,
            pessoa: {
                cpfCnpj: values.cpf
            },
            usuario: {
                nome: values.name,
                email: values.email,
                celular: values.cellPhone,
                senha: values.password,
                perfilAtual: "C",
                linguagemAtual: "pt",
                seAventuras:    userPermission.adventures ? "S" : "N",
                seCategorias:   userPermission.categorys  ? "S" : "N",
                seFornecedores: userPermission.suppliers  ? "S" : "N",
                seMateriais:    userPermission.materials  ? "S" : "N",
                seOverview:     userPermission.overview   ? "S" : "N",
                seUsuarios:     userPermission.users      ? "S" : "N"
            }
        };

        try{
            const response = await createAdminUser(body)
            if(response.status === 200){
                Util.showSucessToast(toast, `Usuário criado com sucesso!`)
                props.onHide()
            }else{
                Util.showErrorToast(toast, `Erro ao criar usuário!`)
            }
        }catch(error){
            Util.showErrorToast(toast, `Erro ao criar usuário!`)
        }
    };

    const PermissionSwitch = ({ text, permission }: any) => {
        return (
            <div style={{ display: "flex", alignItems: "center", padding: "5px" }}>
                {/* @ts-ignore */}
                <InputSwitch style={{ position: "relative", margin: "5px" }}
                    checked={userPermission[permission]}
                    onChange={() => setUserPermission({ ...userPermission, [permission]: !userPermission[permission] })}
                />
                <text>{text}</text>
            </div>
        )
    };

    const initialValues = {
        name: "",
        email: "",
        cellPhone: "",
        cpf: "",
        password: "",
        repeatPassword: "",
    };

    const validationSchema = Yup.object({
        name: Yup.string().required(t("requiredField")),
        email: Yup.string().email(t("invalidEmail")).required(t("requiredField")),
        cellPhone: Yup.string().phone("BR", false, t("invalidCellphone")).required(t("requiredField")),
        cpf: Yup.string().test("cpf inválido", value => {
            if (!value) return false;
            return /^(\d{3}\.){2}\d{3}\-\d{2}$/.test(value) || /^(\d{2}\.)(\d{3}\.)\d{3}\/\d{4}-\d{2}$/.test(value);
        }).required(t("requiredField")),
        password: Yup.string()
            .password()
            .required(t("requiredField"))
            .minUppercase(0)
            .minLowercase(0)
            .minSymbols(0)
            .minNumbers(0)
            .min(6, t("passwordLength"))
            .max(16, t("passwordLength")),
        repeatPassword: Yup.string().when("password", {
            is: (value: any) => value && value.length > 0,
            then: Yup.string()
                .oneOf([Yup.ref("password")], t("passwordsMatch"))
                .required(t("requiredField")),
            otherwise: Yup.string().notRequired(),
        }),
    });

    return (
        <>
        {/* @ts-ignore */}
            <Toast ref={toast} position="top-center" />
            {/* @ts-ignore */}
            <Dialog header="Criação de Usuário" visible={visible} style={{ width: '50vw' }} onHide={() => props.onHide()}>
                <Wrapper>

                    <RegisterImage
                        url={archiveInfos?.uri || "/assets/perfil-default.png"}
                        onClick={() => {
                            {/* @ts-ignore */}
                            inputRef.current.click();
                        }}
                        >
                        <input
                            ref={inputRef}
                            type="file"
                            onChange={(event) => handleOnImageChange(event)}
                        />
                    </RegisterImage>
                    <Formik
                        initialValues={initialValues}
                        validationSchema={validationSchema}
                        onSubmit={(values) => { onSubmit(values) }}
                        validateOnChange
                        >
                        {(formik) => {
                            return (
                                <Form>
                                    <Input
                                        id="name"
                                        name="name"
                                        label={t("name")}
                                        placeholder="João Alberto"
                                        error={formik.errors.name && formik.touched.name}
                                        icon={<UserIcon width={20} height={20}></UserIcon>}
                                        />
                                    <Input
                                        id="email"
                                        name="email"
                                        label={t("email")}
                                        placeholder="example@gmail.com"
                                        error={formik.errors.email && formik.touched.email}
                                        icon={<Mail width={20} height={20}></Mail>}
                                        />
                                    <Input
                                        id="cellPhone"
                                        name="cellPhone"
                                        label={t("cellPhone")}
                                        placeholder="(99) 9 9999-9999"
                                        onInput={(event: any) => {
                                            event.nativeEvent.target.value = masks.brCellphone(event.nativeEvent.target.value);
                                        }}
                                        error={formik.errors.cellPhone && formik.touched.cellPhone}
                                        icon={<Cellphone width={20} height={20}></Cellphone>}
                                        />
                                    <Input
                                        id="cpf"
                                        name="cpf"
                                        label={t("cpf")}
                                        placeholder="999.999.999-99 | 99.999.999/9999-99"
                                        onInput={(event: any) => {
                                            event.nativeEvent.target.value = masks.cpfCnpj(event.nativeEvent.target.value);
                                        }}
                                        error={formik.errors.cpf && formik.touched.cpf}
                                        icon={<Mail width={20} height={20}></Mail>}
                                        />
                                    <Input
                                        id="password"
                                        name="password"
                                        label={t("password")}
                                        type="password"
                                        placeholder="******"
                                        error={formik.errors.password && formik.touched.password}
                                        icon={<Password width={20} height={20}></Password>}
                                        />

                                    <Input
                                        id="repeatPassword"
                                        name="repeatPassword"
                                        label={t("repeatPassword")}
                                        type="password"
                                        placeholder="******"
                                        error={formik.errors.repeatPassword && formik.touched.repeatPassword}
                                        icon={<Password width={20} height={20}></Password>}
                                        />
                                    <PermissionSwitch
                                        text="Vizualizar Overview"
                                        permission="overview"
                                        />
                                    <PermissionSwitch
                                        text="Vizualizar Fornecedores"
                                        permission="suppliers"
                                        />
                                    <PermissionSwitch
                                        text="Vizualizar Usuários"
                                        permission="users"
                                        />
                                    <PermissionSwitch
                                        text="Vizualizar Aventuras"
                                        permission="adventures"
                                        />
                                    <PermissionSwitch
                                        text="Vizualizar materiais"
                                        permission="materials"
                                        />
                                    <PermissionSwitch
                                        text="Vizualizar categorias"
                                        permission="categorys"
                                        />

                                    <ButtonContainer>
                                        <Button type="submit" text={"Criar Usuário"} disabled={!formik.isValid}></Button>
                                    </ButtonContainer>
                                </Form>
                            )
                        }}
                    </Formik>
                </Wrapper>
            </Dialog>
        </>
    )
}