import React, { Dispatch, useReducer, SetStateAction } from "react";
import * as Card from "./styles";
import { AdventureListDto } from "../../../core/models/interfaces/Adventure/AdventureListDto";
import Link from "next/link";
import Heart from "../SvgIcons/heart";
import Star from "../SvgIcons/star";
import FullHeart from "../SvgIcons/fullHeart";
import { FormattedNumber } from "react-intl";
import { useUserContext } from "../../../core/contexts/UserContext";
import WithAuthModal from "../WithAuthModal";
import ElementTag from "../ElementTag";
import useAdventure from "../../../core/hooks/useAdventure";
import { useTranslation } from "next-i18next";

interface CardAdventureProps {
    timers: boolean;
    supplierId: string | number;
    adventure: AdventureListDto;
    setAuthModal: Dispatch<SetStateAction<boolean>>;
}

function CardAdventureUser(props: CardAdventureProps) {
    const [_, forceUpdate] = useReducer((x) => x + 1, 0);
    const { t } = useTranslation("common");
    const { favoriteAdventure, desfavoriteAdventure } = useAdventure();
    const userContext = useUserContext();

    const handleFavoriteOrDesfavorite = async (event: React.MouseEvent) => {
    event.preventDefault();

    if (!userContext.isAuthenticated) {
        props.setAuthModal(true);
        return;
    }

    if (props.adventure.id)
        if (props.adventure.seFavorita === "N") {
        await favoriteAdventure(props.adventure?.id)
            .then(() => {
            props.adventure.seFavorita = "S";
            })
            .catch(() => {});
        } else {
        await desfavoriteAdventure(props.adventure?.id)
            .then(() => {
            props.adventure.seFavorita = "N";
            })
            .catch(() => {});
        }

    forceUpdate();
    };

    const formatarStatus = (status: string) => {
    let paramaters =
        status == "PAID"
        ? ["#4CD964", "Pago"]
        : status == "AUTHORIZED"
        ? ["#4CD964", "Autorizada"]
        : status == "REFUNDED"
        ? ["#4CD964", "Reembolsada"]
        : status == "WAITING_PAYMENT"
        ? ["#FBCB20", "Aguardando pagamento"]
        : status == "PENDING_REFUND"
        ? ["#FBCB20", "Reembolso pendente"]
        : status == "PROCESSING"
        ? ["#FBCB20", "Processando pagamento"]
        : status == "CHARGEDBACK"
        ? ["#D5403B", "Sofreu chargeback"]
        : status == "REFUSED"
        ? ["#D5403B", "Negada"]
        : ["#FBCB20", "Em análise"];
    return (
        <Card.ElementTransAction color={paramaters[0]}>
        { paramaters[1].slice(0, 12) }{paramaters[1].length > 12 ? '...' : ''}
        </Card.ElementTransAction>
    )
    };

    return (
    <>
        <Link href={`/adventure-details/${props.adventure.id}/${props.adventure?.slug}`} passHref>
            <Card.Wrapper>
            <Card.Image url={props.adventure?.urlFoto}>
                <Card.FavoriteContainer onClick={(event) => handleFavoriteOrDesfavorite(event)}>
                {props.adventure?.seFavorita === "S" ? (
                    <FullHeart width={24} height={26} />
                ) : (
                    <Heart fill={"#FFFFFF"} width={24} height={26} />
                )}
            </Card.FavoriteContainer>

            {props.adventure?.avaliacao && !isNaN(props.adventure?.avaliacao) && (
            <Card.RatingContainer>
                <Star />
                <Card.Text>{parseFloat(props.adventure?.avaliacao?.toString()).toFixed(1)}</Card.Text>
            </Card.RatingContainer>
            )}

            {props.adventure.transactionStatus && (
                formatarStatus(props.adventure.transactionStatus)
            )}

            <Card.Element>
                <ElementTag adventureType={props.adventure?.tipo}></ElementTag>
            </Card.Element>
            
            {props.timers && (
                <Card.ButtonAction onClick={() => {
                    location.href = (`/profiles/admin/supplier/${props.supplierId}/schedules-adventures/${props.adventure.id}`);
                }}>
                    Ver horários
                </Card.ButtonAction>
            )}
            </Card.Image>

            <Card.InfoContainer>
                <Card.Title>{props.adventure?.titulo}</Card.Title>
                <Card.Text>
                {t("where")}: {props.adventure?.local.slice(0, 15)}{props.adventure?.local.length > 15 ? '...' : ''}
                </Card.Text>
            </Card.InfoContainer>

            <Card.PriceContainer>
                <Card.Title>
                    <FormattedNumber value={props.adventure?.valor} style="currency" currency="BRL" />
                </Card.Title>
                <Card.Text>{t("perPerson")}</Card.Text>
            </Card.PriceContainer>
                {/* @ts-ignore */}
            {props.adventure?.data && props.adventure?.hora && (
                <Card.InfoAccomplishedContainer>
                    {/* @ts-ignore */}
                    <Card.Text>{props.adventure?.data?.split('-').reverse().join('/')} - {props.adventure?.hora}</Card.Text>
                </Card.InfoAccomplishedContainer>
            )}
            </Card.Wrapper>
        </Link>

        <Card.CardContent align="bottom">
            <div>
                <Card.Title> Informações do fornecedor </Card.Title>
            </div>
            <div>
                <Card.Text> Nome: {props.adventure.fornecedorInformacaoDto.nomeFornecedor} </Card.Text>
            </div>
            <div>
                <Card.Text> Email: {props.adventure.fornecedorInformacaoDto.emailFornecedor} </Card.Text>
            </div>
            <div>
                <Card.Text> Celular: {props.adventure.fornecedorInformacaoDto.telefoneFornecedor} </Card.Text>
            </div>
        </Card.CardContent>
    </>
    );
}

export default React.memo(WithAuthModal(CardAdventureUser));
