import React, { Dispatch, useReducer, SetStateAction, useState, useRef } from 'react';
import * as Card from './styles';
import { AdventureListDto } from '../../../core/models/interfaces/Adventure/AdventureListDto';
import Link from 'next/link';
import Heart from '../SvgIcons/heart';
import Star from '../SvgIcons/star';
import FullHeart from '../SvgIcons/fullHeart';
import { FormattedNumber } from 'react-intl';
import { useUserContext } from '../../../core/contexts/UserContext';
import WithAuthModal from '../WithAuthModal';
import ElementTag from '../ElementTag';
import useAdventure from '../../../core/hooks/useAdventure';
import { useTranslation } from 'next-i18next';
import { Dialog } from "primereact/dialog"
import Button from "../Button";
import { Toast } from "primereact/toast";
import { Util } from "../../../core/utils/util";
import router from 'next/router';

interface CardAdventureProps {
  timers: boolean;
  supplierId: string | number;
  adventure: AdventureListDto;
  setAuthModal: Dispatch<SetStateAction<boolean>>;
  isEditCrew?: boolean;
}

function CardAdventure({
  timers,
  supplierId,
  adventure,
  setAuthModal,
  isEditCrew = false,
}: CardAdventureProps) {
  const [_, forceUpdate] = useReducer((x) => x + 1, 0);
  const { t } = useTranslation('common');
  const { favoriteAdventure, desfavoriteAdventure, deleteDataAventure } = useAdventure();
  const userContext = useUserContext();
  const [visibleModalDelete, setVisibleModalDelete] = useState(false)
  const toast = useRef(null);

  const handleFavoriteOrDesfavorite = async (event: React.MouseEvent) => {
    event.preventDefault();

    if (!userContext.isAuthenticated) {
      setAuthModal(true);
      return;
    }

    if (adventure.id)
      if (adventure.seFavorita === 'N') {
        await favoriteAdventure(adventure?.id)
          .then(() => {
            adventure.seFavorita = 'S';
          })
          .catch(() => {});
      } else {
        await desfavoriteAdventure(adventure?.id)
          .then(() => {
            adventure.seFavorita = 'N';
          })
          .catch(() => {});
      }

    forceUpdate();
  };

  const deleteAdventureOk = async (adventureId: number) => {
    const response = await deleteDataAventure(adventureId)
    if(response.status === 200){
      setVisibleModalDelete(!visibleModalDelete)
      Util.showSucessToast(toast, "Aventura excluida com sucesso");
      location.reload()
    } else {
      Util.showErrorToast(toast, "Erro ao excluir aventura");
    }
  }

  return (
    <>
    {/* @ts-ignore */}
      <Toast ref={toast} position="top-center" />
      {/* Modal for delete adventure  */}
      {/* @ts-ignore */}
      <Dialog header="Atenção" visible={visibleModalDelete} style={{ width: '50vw' }} onHide={() => setVisibleModalDelete(!visibleModalDelete) }>
        <div>
          Deseja excluir aventura selecionada ? 
        </div>
        <br />
        <Card.InputGroup inputs={2}>
          <Button type="button" color="#495057a3" onClick={() => setVisibleModalDelete(!visibleModalDelete) } text="Não"></Button>
          {/* @ts-ignore */}
          <Button type="button" text="Sim" onClick={() => deleteAdventureOk(adventure.id)}/>
        </Card.InputGroup>
      </Dialog>

      {isEditCrew && (
        <Card.ButtonGroup>
          <Card.ButtonConfig onClick={() => setVisibleModalDelete(!visibleModalDelete) }>Excluir</Card.ButtonConfig>
          <Card.ButtonConfig
            onClick={() =>
              (location.href = `/profiles/supplier/crews/${adventure.id}`)
            }
          >
            Turmas
          </Card.ButtonConfig>
          <Card.ButtonConfig
            onClick={() =>
              (location.href = `/profiles/supplier/adventures/edit/${adventure.id}`)
            }
          >
            Editar
          </Card.ButtonConfig>
        </Card.ButtonGroup>
      )}

      <Link
        href={`/adventure-details/${adventure?.id}/${adventure?.slug}`}
        passHref
      >
        <Card.Wrapper>
          <Card.Image 
          url={adventure?.urlFoto}
          onClick={() => router.push(`/adventure-details/${adventure?.id}/${adventure?.slug}`)}
          >
            <Card.FavoriteContainer
              onClick={(event) => handleFavoriteOrDesfavorite(event)}
            >
              {adventure?.seFavorita === 'S' ? (
                <FullHeart width={24} height={26} />
              ) : (
                <Heart fill={'#FFFFFF'} width={24} height={26} />
              )}
            </Card.FavoriteContainer>
            {adventure?.avaliacao && !isNaN(adventure?.avaliacao) && (
              <Card.RatingContainer>
                <Star />
                <Card.Text>
                  {parseFloat(adventure?.avaliacao?.toString()).toFixed(1)}
                </Card.Text>
              </Card.RatingContainer>
            )}

            <Card.Element>
              <ElementTag adventureType={adventure?.tipo}></ElementTag>
            </Card.Element>

            <Card.Element>
              <ElementTag adventureType={adventure?.tipo}></ElementTag>
            </Card.Element>
          </Card.Image>

          {timers && (
            <Card.ButtonAction
              onClick={() => {
                location.href = `/profiles/admin/supplier/${supplierId}/schedules-adventures/${adventure.id}`;
              }}
            >
              Ver horários
            </Card.ButtonAction>
          )}

          <Card.InfoContainer>
            <Card.Title>{adventure?.titulo}</Card.Title>
            <Card.Text>
              {t('where')}: {adventure?.local}
            </Card.Text>
          </Card.InfoContainer>

          <Card.PriceContainer>
            <Card.Title>
              <FormattedNumber
                value={adventure?.valor}
                style="currency"
                currency="BRL"
              />
            </Card.Title>
            <Card.Text>{t('perPerson')}</Card.Text>
          </Card.PriceContainer>
{/* @ts-ignore */}
          {adventure?.data && adventure?.hora && (
            <Card.InfoAccomplishedContainer>
              <Card.Text>
                {/* @ts-ignore */}
                {adventure?.data?.split('-').reverse().join('/')} - {' '}
                {/* @ts-ignore */}
                {adventure?.hora}
              </Card.Text>
            </Card.InfoAccomplishedContainer>
          )}
        </Card.Wrapper>
      </Link>
    </>
  );
}

export default React.memo(WithAuthModal(CardAdventure));
